<?php
// compare.php - MLS9 Deal Comparison Page
$page_title = 'Compare Properties';
$activePage = 'compare'; // make sure header_nav/sidebar use this the same way as other pages

include __DIR__ . '/header_nav.php';
?>

<div class="app-shell"><!-- if your header_nav already outputs this wrapper, remove this line -->
    <?php include __DIR__ . '/sidebar_property.php'; ?>

    <div class="main">
        <header class="topbar">
            <div class="topbar-title">
                <h1><i class="fa-solid fa-columns"></i>Compare Properties</h1>
                <span>Side-by-side comparison of up to 4 deals using full MLS9 & RentCast data.</span>
            </div>
            <div class="topbar-actions">
                <div class="badge-pill">
                    <i class="fa-solid fa-cloud-arrow-down"></i>
                    Data cached from RentCast & other sources to reduce API costs.
                </div>
            </div>
        </header>

        <main class="content">
            <div class="compare-layout">
                <!-- Toolbar -->
                <section class="compare-toolbar">
                    <div class="compare-toolbar-left">
                        <div class="input-group">
                            <input id="address-input"
                                   type="text"
                                   placeholder="Enter property address to add to comparison…" />
                            <button id="add-property-btn" type="button">
                                <i class="fa-solid fa-plus"></i> Add Property
                            </button>
                        </div>
                    </div>

                    <div class="compare-toolbar-right">
                        <span class="small-label">Sort by</span>
                        <select id="sort-metric-select" class="select-compact">
                            <option value="cocReturn">Cash-on-Cash Return</option>
                            <option value="cashFlowMonthly">Monthly Cash Flow</option>
                            <option value="capRate">Cap Rate</option>
                            <option value="estimatedValue">Estimated Value</option>
                            <option value="rentEstimate">Rent Estimate</option>
                        </select>

                        <button id="clear-all-btn" type="button" class="btn btn-ghost">
                            <i class="fa-solid fa-trash-can"></i> Clear All
                        </button>
                    </div>
                </section>

                <div class="selected-properties" id="selected-properties"></div>
                <div id="error-banner" class="error-banner"></div>

                <!-- Compare card -->
                <section class="compare-card">
                    <div class="compare-card-header">
                        <div class="compare-card-header-left">
                            <h2><i class="fa-solid fa-scale-balanced"></i>DealCheck Comparison</h2>
                            <p>Overview, valuations, income, expenses, and returns for each property.</p>
                        </div>
                        <div class="compare-card-header-right">
                            <div id="loading-indicator" class="loading-indicator">
                                <span class="spinner"></span>
                                <span>Analyzing property… pulling RentCast & cached MLS9 data.</span>
                            </div>
                        </div>
                    </div>

                    <div class="compare-scroll-wrapper">
                        <table class="compare-table" id="compare-table">
                            <!-- Content rendered by JS -->
                        </table>
                    </div>

                    <div class="compare-footer-note">
                        <span><i class="fa-solid fa-circle-info"></i> All metrics are based on your stored assumptions from the Purchase & Analysis pages.</span>
                        <span>Best deals float to the left when you change the sort metric.</span>
                    </div>
                </section>
            </div>
        </main>
    </div>
</div><!-- if header_nav opens app-shell, remove this closing tag too -->

<?php include __DIR__ . '/footer.php'; ?>

<script>
    const MAX_COMPARE_PROPERTIES = 4;
    const PROPERTY_ENDPOINT = 'property.php'; // central analyzer

    let compareItems = [];

    const addressInput = document.getElementById('address-input');
    const addPropertyBtn = document.getElementById('add-property-btn');
    const selectedPropertiesContainer = document.getElementById('selected-properties');
    const compareTable = document.getElementById('compare-table');
    const loadingIndicator = document.getElementById('loading-indicator');
    const errorBanner = document.getElementById('error-banner');
    const sortMetricSelect = document.getElementById('sort-metric-select');
    const clearAllBtn = document.getElementById('clear-all-btn');

    function showLoading(show) {
        if (!loadingIndicator) return;
        loadingIndicator.style.display = show ? 'flex' : 'none';
    }

    function showError(message) {
        if (!errorBanner) return;
        if (!message) {
            errorBanner.style.display = 'none';
            errorBanner.textContent = '';
            return;
        }
        errorBanner.textContent = message;
        errorBanner.style.display = 'block';
    }

    function formatCurrency(value) {
        if (value === null || value === undefined || isNaN(value)) return '–';
        return '$' + Number(value).toLocaleString(undefined, { maximumFractionDigits: 0 });
    }

    function formatNumber(value, decimals = 0) {
        if (value === null || value === undefined || isNaN(value)) return '–';
        return Number(value).toLocaleString(undefined, { maximumFractionDigits: decimals });
    }

    function formatPercent(value, decimals = 1) {
        if (value === null || value === undefined || isNaN(value)) return '–';
        return Number(value).toFixed(decimals) + '%';
    }

    function clamp(value, min, max) {
        return Math.min(max, Math.max(min, value));
    }

    function normalizePropertyData(raw) {
        const d = raw && raw.data ? raw.data : {};

        const addressLine = d.address || '';
        const city = d.city || d.cityName || '';
        const state = d.state || d.stateCode || '';
        const zip = d.zip || d.zipCode || '';

        const valuations = d.valuations || {};
        const rent = d.rent || {};
        const fin = d.financials || {};
        const risk = d.risk || {};

        return {
            id: d.id || (addressLine + '|' + city + '|' + state + '|' + zip),
            rawAddress: addressLine,
            displayAddress: addressLine,
            city: city,
            state: state,
            zip: zip,
            beds: d.bedrooms || d.beds || null,
            baths: d.bathrooms || d.baths || null,
            squareFeet: d.squareFeet || d.sqft || null,
            lotSize: d.lotSize || d.lotSizeSqft || null,
            propertyType: d.propertyType || d.type || '',
            yearBuilt: d.yearBuilt || null,
            metrics: {
                estimatedValue: valuations.estimatedValue || valuations.value || d.estimatedValue || null,
                valueLow: valuations.low || valuations.min || null,
                valueHigh: valuations.high || valuations.max || null,
                rentEstimate: rent.estimate || rent.rentEstimate || null,
                rentLow: rent.low || rent.min || null,
                rentHigh: rent.high || rent.max || null,
                monthlyRent: fin.monthlyRent || d.monthlyRent || rent.estimate || null,
                otherIncome: fin.otherIncome || null,
                operatingExpenses: fin.operatingExpenses || fin.expenses || null,
                noiAnnual: fin.noiAnnual || d.noiAnnual || null,
                cashFlowMonthly: fin.cashFlowMonthly || d.cashFlowMonthly || null,
                capRate: fin.capRate || d.capRate || null,
                cocReturn: fin.cocReturn || d.cocReturn || null,
                totalReturn5yr: fin.totalReturn5yr || d.totalReturn5yr || null,
                vacancyRate: risk.vacancyRate || d.vacancyRate || null,
                neighborhoodScore: risk.neighborhoodScore || d.neighborhoodScore || null,
                riskScore: risk.riskScore || d.riskScore || null,
                appreciation5yr: risk.appreciation5yr || d.appreciation5yr || null
            }
        };
    }

    function computeDealRating(metrics) {
        const cashFlow = Number(metrics.cashFlowMonthly || 0);
        const coc = Number(metrics.cocReturn || 0);
        const cap = Number(metrics.capRate || 0);

        let score = 0;
        score += clamp(cashFlow / 200, -2, 4);
        score += clamp(coc / 5, -2, 4);
        score += clamp(cap / 4, -1, 3);

        let label = 'Neutral';
        let icon = 'fa-circle-half-stroke';
        let className = '';

        if (score >= 6) {
            label = 'Great Deal';
            icon = 'fa-face-laugh-beam';
        } else if (score <= 1) {
            label = 'Risky / Thin';
            icon = 'fa-triangle-exclamation';
            className = 'bad';
        }

        return { score, label, icon, className };
    }

    function renderSelectedChips() {
        if (!selectedPropertiesContainer) return;
        selectedPropertiesContainer.innerHTML = '';
        if (!compareItems.length) return;

        compareItems.forEach((item, index) => {
            const chip = document.createElement('div');
            chip.className = 'property-chip';
            chip.innerHTML = `
                <span>${item.displayAddress || item.rawAddress || 'Property ' + (index + 1)}</span>
                <button type="button" data-id="${item.id}">
                    <i class="fa-solid fa-xmark"></i>
                </button>
            `;
            selectedPropertiesContainer.appendChild(chip);
        });

        selectedPropertiesContainer.querySelectorAll('button[data-id]').forEach(btn => {
            btn.addEventListener('click', () => {
                const id = btn.getAttribute('data-id');
                removePropertyById(id);
            });
        });
    }

    function renderTable() {
        if (!compareTable) return;

        if (!compareItems.length) {
            compareTable.innerHTML = `
                <tbody>
                    <tr>
                        <td class="empty-state" colspan="5">
                            Add up to ${MAX_COMPARE_PROPERTIES} properties above to compare them side-by-side like DealCheck.
                        </td>
                    </tr>
                </tbody>
            `;
            return;
        }

        const headerCols = compareItems.map(item => {
            const m = item.metrics || {};
            const rating = computeDealRating(m);
            const cityState = [item.city, item.state].filter(Boolean).join(', ');

            return `
                <th>
                    <div class="property-col-header">
                        <div class="property-col-header-title">${item.displayAddress || item.rawAddress || 'Property'}</div>
                        <div class="property-col-header-sub">${cityState} ${item.zip || ''}</div>
                        <div class="property-col-header-sub">
                            ${m.estimatedValue ? formatCurrency(m.estimatedValue) + ' est. value' : ''}
                        </div>
                        <div class="property-col-header-sub">
                            <span class="pill-rating ${rating.className}">
                                <i class="fa-solid ${rating.icon}"></i> ${rating.label}
                            </span>
                        </div>
                    </div>
                </th>
            `;
        }).join('');

        function row(label, hint, key, formatFn, cssClass = 'metric-neutral') {
            const cells = compareItems.map(item => {
                const m = item.metrics || {};
                const value = m[key];
                let formatted = formatFn(value);

                let valueClass = cssClass;
                if (key === 'cashFlowMonthly') {
                    if (value > 0) valueClass = 'metric-positive';
                    else if (value < 0) valueClass = 'metric-negative';
                }
                if (key === 'cocReturn' || key === 'capRate') {
                    if (value >= 10) valueClass = 'metric-positive';
                    else if (value < 0) valueClass = 'metric-negative';
                }

                return `
                    <td>
                        <div class="metric-value ${valueClass}">${formatted}</div>
                    </td>
                `;
            }).join('');

            return `
                <tr>
                    <td class="compare-label-col">
                        <div class="metric-label">${label}</div>
                        ${hint ? `<div class="metric-hint">${hint}</div>` : ''}
                    </td>
                    ${cells}
                </tr>
            `;
        }

        const overviewSection = `
            <tr class="section-row">
                <td class="compare-label-col" colspan="${compareItems.length + 1}">Property Overview</td>
            </tr>
        `;

        const overviewDetails = (() => {
            const rows = [];

            const bedsRow = compareItems.map(item => `
                <td>
                    <div class="metric-value metric-neutral">
                        ${item.beds != null ? formatNumber(item.beds, 0) + ' bd' : '–'}
                        ${item.baths != null ? ' / ' + formatNumber(item.baths, 0) + ' ba' : ''}
                    </div>
                    <div class="metric-sub">
                        ${item.squareFeet ? formatNumber(item.squareFeet) + ' sqft' : ''}
                    </div>
                </td>
            `).join('');

            rows.push(`
                <tr>
                    <td class="compare-label-col">
                        <div class="metric-label">Beds / Baths</div>
                        <div class="metric-hint">Interior square footage from property data.</div>
                    </td>
                    ${bedsRow}
                </tr>
            `);

            const lotRow = compareItems.map(item => `
                <td>
                    <div class="metric-value metric-neutral">
                        ${item.lotSize ? formatNumber(item.lotSize) + ' sqft' : '–'}
                    </div>
                </td>
            `).join('');

            rows.push(`
                <tr>
                    <td class="compare-label-col">
                        <div class="metric-label">Lot Size</div>
                        <div class="metric-hint">Total land area from public records.</div>
                    </td>
                    ${lotRow}
                </tr>
            `);

            const typeRow = compareItems.map(item => `
                <td>
                    <div class="metric-value metric-neutral">
                        ${item.propertyType || '—'}
                    </div>
                    <div class="metric-sub">
                        ${item.yearBuilt ? 'Built ' + item.yearBuilt : ''}
                    </div>
                </td>
            `).join('');

            rows.push(`
                <tr>
                    <td class="compare-label-col">
                        <div class="metric-label">Type / Year</div>
                        <div class="metric-hint">Property classification and year built.</div>
                    </td>
                    ${typeRow}
                </tr>
            `);

            return rows.join('');
        })();

        const valuationsSection = `
            <tr class="section-row">
                <td class="compare-label-col" colspan="${compareItems.length + 1}">Valuations</td>
            </tr>
            ${row('Estimated Value', 'Blended estimate from all sources.', 'estimatedValue', formatCurrency)}
        `;

        const valueRangeRows = (() => {
            const cells = compareItems.map(item => {
                const m = item.metrics || {};
                const low = m.valueLow;
                const high = m.valueHigh;
                const hasRange = (low != null && !isNaN(low)) || (high != null && !isNaN(high));

                if (!hasRange) {
                    return `<td><div class="metric-value metric-neutral">–</div></td>`;
                }

                return `
                    <td>
                        <div class="metric-value metric-neutral">
                            ${low ? formatCurrency(low) : '–'} – ${high ? formatCurrency(high) : '–'}
                        </div>
                    </td>
                `;
            }).join('');

            return `
                <tr>
                    <td class="compare-label-col">
                        <div class="metric-label">Value Range</div>
                        <div class="metric-hint">Low / high confidence band.</div>
                    </td>
                    ${cells}
                </tr>
            `;
        })();

        const rentSection = `
            ${row('Rent Estimate', 'Market rent estimate for this property.', 'rentEstimate', formatCurrency)}
            ${(() => {
                const cells = compareItems.map(item => {
                    const m = item.metrics || {};
                    const low = m.rentLow;
                    const high = m.rentHigh;
                    const hasRange = (low != null && !isNaN(low)) || (high != null && !isNaN(high));

                    if (!hasRange) {
                        return `<td><div class="metric-value metric-neutral">–</div></td>`;
                    }

                    return `
                        <td>
                            <div class="metric-value metric-neutral">
                                ${low ? formatCurrency(low) : '–'} – ${high ? formatCurrency(high) : '–'}
                            </div>
                        </td>
                    `;
                }).join('');

                return `
                    <tr>
                        <td class="compare-label-col">
                            <div class="metric-label">Rent Range</div>
                            <div class="metric-hint">High / low rent band.</div>
                        </td>
                        ${cells}
                    </tr>
                `;
            })()}
        `;

        const incomeSection = `
            <tr class="section-row">
                <td class="compare-label-col" colspan="${compareItems.length + 1}">Income & Expenses</td>
            </tr>
            ${row('Monthly Rent', 'Scheduled base rent for long-term hold.', 'monthlyRent', formatCurrency)}
            ${row('Other Income', 'Parking, pet fees, utilities, etc.', 'otherIncome', formatCurrency)}
            ${row('Operating Expenses', 'Taxes, insurance, repairs, mgmt, etc.', 'operatingExpenses', formatCurrency)}
            ${row('NOI (Annual)', 'Net Operating Income before financing.', 'noiAnnual', formatCurrency)}
        `;

        const returnsSection = `
            <tr class="section-row">
                <td class="compare-label-col" colspan="${compareItems.length + 1}">Return Metrics</td>
            </tr>
            ${row('Monthly Cash Flow', 'Net after all expenses & financing.', 'cashFlowMonthly', formatCurrency)}
            ${row('Cap Rate', 'NOI / Purchase Price.', 'capRate', (v) => formatPercent(v, 1))}
            ${row('Cash-on-Cash', 'Annual cash return on cash invested.', 'cocReturn', (v) => formatPercent(v, 1))}
            ${row('Total Return (5yr)', 'Projected total return over 5-year hold.', 'totalReturn5yr', (v) => formatPercent(v, 0))}
        `;

        const riskSection = `
            <tr class="section-row">
                <td class="compare-label-col" colspan="${compareItems.length + 1}">Risk & Market</td>
            </tr>
            ${row('Vacancy Rate', 'Estimated market vacancy rate.', 'vacancyRate', (v) => formatPercent(v, 1))}
            ${row('Neighborhood Score', 'Quality & desirability 1-100.', 'neighborhoodScore', (v) => v != null ? formatNumber(v, 0) + '/100' : '–')}
            ${row('Deal Risk Score', 'Composite risk indicator (lower is better).', 'riskScore', (v) => v != null ? formatNumber(v, 0) + '/100' : '–')}
            ${row('Appreciation (5yr)', 'Projected value growth.', 'appreciation5yr', (v) => formatPercent(v, 0))}
        `;

        compareTable.innerHTML = `
            <thead>
                <tr>
                    <th class="compare-label-col">Metric</th>
                    ${headerCols}
                </tr>
            </thead>
            <tbody>
                ${overviewSection}
                ${overviewDetails}
                ${valuationsSection}
                ${valueRangeRows}
                ${rentSection}
                ${incomeSection}
                ${returnsSection}
                ${riskSection}
            </tbody>
        `;
    }

    function sortCompareItems(metricKey) {
        compareItems.sort((a, b) => {
            const am = a.metrics || {};
            const bm = b.metrics || {};
            const av = Number(am[metricKey] || 0);
            const bv = Number(bm[metricKey] || 0);
            return bv - av;
        });
    }

    function removePropertyById(id) {
        compareItems = compareItems.filter(item => item.id !== id);
        renderSelectedChips();
        renderTable();
    }

    async function fetchPropertyData(address) {
        showLoading(true);
        showError('');

        try {
            const formData = new FormData();
            formData.append('mode', 'analyze'); // adjust if property.php expects different keys
            formData.append('address', address);

            const response = await fetch(PROPERTY_ENDPOINT, {
                method: 'POST',
                body: formData
            });

            if (!response.ok) {
                throw new Error('Server returned ' + response.status);
            }

            const json = await response.json();

            if (!json || json.success === false) {
                throw new Error(json && json.message ? json.message : 'Unable to analyze property.');
            }

            return normalizePropertyData(json);
        } catch (err) {
            console.error('Error loading property:', err);
            showError('Error loading property: ' + err.message);
            return null;
        } finally {
            showLoading(false);
        }
    }

    async function handleAddProperty() {
        const addr = addressInput.value.trim();
        if (!addr) return;

        if (compareItems.length >= MAX_COMPARE_PROPERTIES) {
            showError('You can compare up to ' + MAX_COMPARE_PROPERTIES + ' properties at a time.');
            return;
        }

        const exists = compareItems.some(item =>
            (item.rawAddress || '').toLowerCase() === addr.toLowerCase()
        );
        if (exists) {
            showError('That property is already in your comparison.');
            return;
        }

        const normalized = await fetchPropertyData(addr);
        if (!normalized) return;

        compareItems.push(normalized);

        const sortKey = sortMetricSelect.value || 'cocReturn';
        sortCompareItems(sortKey);

        addressInput.value = '';
        renderSelectedChips();
        renderTable();
    }

    if (addPropertyBtn) {
        addPropertyBtn.addEventListener('click', handleAddProperty);
    }

    if (addressInput) {
        addressInput.addEventListener('keydown', (e) => {
            if (e.key === 'Enter') {
                e.preventDefault();
                handleAddProperty();
            }
        });
    }

    if (sortMetricSelect) {
        sortMetricSelect.addEventListener('change', () => {
            const sortKey = sortMetricSelect.value || 'cocReturn';
            sortCompareItems(sortKey);
            renderTable();
        });
    }

    if (clearAllBtn) {
        clearAllBtn.addEventListener('click', () => {
            compareItems = [];
            showError('');
            renderSelectedChips();
            renderTable();
        });
    }

    renderTable();
</script>
