<?php
// map.php
session_start();
require_once __DIR__ . '/config.php'; // must define $pdo (PDO) and $API_KEYS

// ---- Helpers ----
function getPropertyById(PDO $pdo, int $id): ?array {
    $stmt = $pdo->prepare("
        SELECT id, full_address, address_line1, address_line2,
               city, state, postal_code, latitude, longitude
        FROM properties
        WHERE id = :id
        LIMIT 1
    ");
    $stmt->execute([':id' => $id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ?: null;
}

// ---- Get property_id ----
$propertyId = isset($_GET['property_id']) ? (int)$_GET['property_id'] : 0;
if ($propertyId <= 0) {
    http_response_code(400);
    echo "Missing or invalid property_id.";
    exit;
}

$property = getPropertyById($pdo, $propertyId);
if (!$property) {
    http_response_code(404);
    echo "Property not found.";
    exit;
}

// Build a nice display address
$displayAddress = $property['full_address'] ??
    trim(
        ($property['address_line1'] ?? '') . ' ' .
        ($property['address_line2'] ?? '') . ', ' .
        ($property['city'] ?? '') . ', ' .
        ($property['state'] ?? '') . ' ' .
        ($property['postal_code'] ?? '')
    );

$lat = $property['latitude'] ?? null;
$lng = $property['longitude'] ?? null;

// Google Maps JS key from config
$googleMapsKey = '';
if (!empty($API_KEYS['google_maps']['enabled']) && !empty($API_KEYS['google_maps']['key'])) {
    $googleMapsKey = $API_KEYS['google_maps']['key'];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Map – <?= htmlspecialchars($displayAddress ?: 'Property #' . $propertyId) ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <!-- Page styles (can be replaced with your global CSS) -->
    <style>
        * { box-sizing: border-box; }
        body {
            margin: 0;
            font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            background: #f3f4f6;
            color: #111827;
        }
        header {
            background: #111827;
            color: #f9fafb;
            padding: 14px 24px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        header h1 {
            margin: 0;
            font-size: 1.1rem;
            font-weight: 600;
        }
        header .breadcrumbs {
            font-size: .85rem;
            color: #9ca3af;
        }
        .layout {
            display: flex;
            min-height: calc(100vh - 56px);
        }
        nav.sidebar {
            width: 230px;
            background: #111827;
            color: #9ca3af;
            padding: 16px 12px;
        }
        nav.sidebar h2 {
            font-size: .75rem;
            text-transform: uppercase;
            letter-spacing: .08em;
            margin: 0 0 8px;
            color: #6b7280;
        }
        nav.sidebar a {
            display: block;
            padding: 8px 10px;
            border-radius: 8px;
            text-decoration: none;
            color: inherit;
            font-size: .9rem;
        }
        nav.sidebar a.active {
            background: #1f2937;
            color: #e5e7eb;
            font-weight: 500;
        }
        nav.sidebar a:hover {
            background: #1f2937;
            color: #e5e7eb;
        }
        .content {
            flex: 1;
            padding: 20px 24px 32px;
            overflow-x: hidden;
            display: flex;
            flex-direction: column;
            gap: 16px;
        }
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            gap: 16px;
        }
        .page-header-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: #111827;
        }
        .page-header-subtitle {
            font-size: .9rem;
            color: #6b7280;
        }
        .pill {
            display: inline-flex;
            align-items: center;
            padding: 4px 10px;
            border-radius: 999px;
            font-size: .75rem;
            background: #e5e7eb;
            color: #374151;
        }
        .pill span.dot {
            width: 6px;
            height: 6px;
            border-radius: 999px;
            background: #3b82f6;
            margin-right: 6px;
        }
        .main-grid {
            display: grid;
            grid-template-columns: minmax(0, 3fr) minmax(0, 1.5fr);
            gap: 16px;
            min-height: 0;
            flex: 1;
        }
        .card {
            background: #ffffff;
            border-radius: 14px;
            padding: 14px 16px 16px;
            box-shadow: 0 10px 25px rgba(15,23,42,0.06);
            border: 1px solid #e5e7eb;
        }
        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 10px;
        }
        .card-header h3 {
            margin: 0;
            font-size: 0.95rem;
            font-weight: 600;
            color: #111827;
        }
        .card-header span {
            font-size: .8rem;
            color: #6b7280;
        }
        #map {
            width: 100%;
            height: 440px;
            border-radius: 12px;
            border: 1px solid #e5e7eb;
        }
        .map-controls {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
            margin-bottom: 10px;
            font-size: .8rem;
        }
        .btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            padding: 6px 10px;
            border-radius: 999px;
            border: 1px solid #d1d5db;
            background: #ffffff;
            color: #111827;
            font-size: .8rem;
            cursor: pointer;
        }
        .btn.active {
            background: #111827;
            color: #f9fafb;
            border-color: #111827;
        }
        .btn:disabled {
            opacity: 0.6;
            cursor: default;
        }
        .property-summary {
            font-size: .9rem;
            color: #4b5563;
            display: grid;
            grid-template-columns: minmax(0,1.3fr) minmax(0,1fr);
            gap: 8px 14px;
        }
        .property-summary dt {
            font-weight: 500;
            color: #111827;
        }
        .property-summary dd {
            margin: 0;
        }
        .chip-row {
            display: flex;
            flex-wrap: wrap;
            gap: 6px;
            margin-top: 6px;
        }
        .chip {
            padding: 3px 8px;
            border-radius: 999px;
            background: #eff6ff;
            color: #1d4ed8;
            font-size: .75rem;
        }
        .alert {
            border-radius: 999px;
            padding: 6px 10px;
            font-size: .75rem;
            margin-top: 10px;
        }
        .alert-warning {
            background: #fffbeb;
            color: #92400e;
            border: 1px solid #fed7aa;
        }
        @media (max-width: 960px) {
            .layout {
                flex-direction: column;
            }
            nav.sidebar {
                width: 100%;
                display: flex;
                flex-wrap: wrap;
                gap: 4px;
            }
            nav.sidebar a {
                display: inline-flex;
            }
            .main-grid {
                grid-template-columns: minmax(0, 1fr);
            }
            #map {
                height: 320px;
            }
        }
    </style>
</head>
<body>
<header>
    <div>
        <div class="breadcrumbs">MLS9 &nbsp;•&nbsp; Map</div>
        <h1>
            Map – <?= htmlspecialchars($displayAddress ?: 'Property #' . $propertyId) ?>
        </h1>
    </div>
    <div class="pill">
        <span class="dot"></span>
        Property ID #<?= (int)$propertyId ?>
    </div>
</header>

<div class="layout">

    <!-- Sidebar navigation (same structure as other pages) -->
    <nav class="sidebar">
        <h2>Property</h2>
        <a href="property.php?property_id=<?= (int)$propertyId ?>">Overview</a>
        <a href="purchase.html?property_id=<?= (int)$propertyId ?>">Purchase</a>
        <a href="photos.php?property_id=<?= (int)$propertyId ?>">Photos</a>
        <a href="map.php?property_id=<?= (int)$propertyId ?>" class="active">Map</a>
        <a href="analysis.html?property_id=<?= (int)$propertyId ?>">Analysis</a>
        <a href="projections.html?property_id=<?= (int)$propertyId ?>">Projections</a>
        <a href="sales-comps.html?property_id=<?= (int)$propertyId ?>">Sales Comps</a>
        <a href="rental-comps.html?property_id=<?= (int)$propertyId ?>">Rental Comps</a>
        <a href="records.html?property_id=<?= (int)$propertyId ?>">Records</a>
        <a href="owner-lookup.html?property_id=<?= (int)$propertyId ?>">Owner Lookup</a>
        <a href="offer-calculator.html?property_id=<?= (int)$propertyId ?>">Offer Calculator</a>
        <a href="reports.html?property_id=<?= (int)$propertyId ?>">Reports</a>
        <a href="compare.html?property_id=<?= (int)$propertyId ?>">Compare</a>
        <a href="lenders.html?property_id=<?= (int)$propertyId ?>">Lenders</a>
        <a href="track.html?property_id=<?= (int)$propertyId ?>">Track</a>
    </nav>

    <!-- Main content -->
    <main class="content">
        <div class="page-header">
            <div>
                <div class="page-header-title">Location & Surroundings</div>
                <div class="page-header-subtitle">
                    See where this property sits on the map, toggle views, and verify that the area
                    matches your deal assumptions.
                </div>
            </div>
        </div>

        <div class="main-grid">
            <!-- Left: Map -->
            <section class="card">
                <div class="card-header">
                    <h3>Interactive Map</h3>
                    <span><?= htmlspecialchars($displayAddress ?: 'No address on file') ?></span>
                </div>

                <?php if (!$googleMapsKey): ?>
                    <div class="alert alert-warning">
                        Google Maps is not configured. Add your key to <code>$API_KEYS['google_maps']</code>
                        in <code>config.php</code> to enable the map.
                    </div>
                <?php endif; ?>

                <div class="map-controls">
                    <button type="button" class="btn active" data-map-type="roadmap">Road</button>
                    <button type="button" class="btn" data-map-type="satellite">Satellite</button>
                    <button type="button" class="btn" id="btn-toggle-streetview">Street View</button>
                    <button type="button" class="btn" id="btn-center-property">Center on Property</button>
                </div>

                <div id="map"></div>

                <div style="font-size: .75rem; color: #9ca3af; margin-top: 8px;">
                    Tip: Use this view alongside your comps and projections to sanity-check
                    neighborhood quality, road access, and nearby amenities.
                </div>
            </section>

            <!-- Right: Property + coordinates -->
            <section class="card">
                <div class="card-header">
                    <h3>Location Details</h3>
                    <span>Quick reference</span>
                </div>

                <dl class="property-summary">
                    <dt>Address</dt>
                    <dd><?= htmlspecialchars($displayAddress ?: 'N/A') ?></dd>

                    <dt>City / State</dt>
                    <dd>
                        <?= htmlspecialchars(($property['city'] ?? '') . ', ' . ($property['state'] ?? '')) ?>
                    </dd>

                    <dt>ZIP</dt>
                    <dd><?= htmlspecialchars($property['postal_code'] ?? 'N/A') ?></dd>

                    <dt>Coordinates</dt>
                    <dd>
                        <?php if ($lat && $lng): ?>
                            <?= htmlspecialchars($lat) ?>, <?= htmlspecialchars($lng) ?>
                        <?php else: ?>
                            Not stored – will geocode from address.
                        <?php endif; ?>
                    </dd>
                </dl>

                <div class="chip-row" style="margin-top: 10px;">
                    <span class="chip">🔍 Validate neighborhood</span>
                    <span class="chip">🧮 Cross-check comps</span>
                    <span class="chip">🚗 Check access</span>
                </div>

                <?php if (!$lat || !$lng): ?>
                    <div class="alert alert-warning">
                        We don’t have latitude/longitude saved for this property yet. The map will
                        attempt to geocode using the address. Once you’re happy with the location,
                        you can write a small endpoint to save the coordinates back into
                        <code>properties.latitude</code> and <code>properties.longitude</code>
                        for faster future loads.
                    </div>
                <?php endif; ?>
            </section>
        </div>
    </main>
</div>

<?php if ($googleMapsKey): ?>
<script>
    // Pass property data from PHP to JS
    const PROPERTY = <?= json_encode([
        'id'      => (int)$property['id'],
        'address' => $displayAddress,
        'lat'     => $lat !== null ? (float)$lat : null,
        'lng'     => $lng !== null ? (float)$lng : null,
    ], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) ?>;
</script>
<script async
    defer
    src="https://maps.googleapis.com/maps/api/js?key=<?= htmlspecialchars($googleMapsKey) ?>&libraries=places&callback=initMap">
</script>
<script>
    let map;
    let propertyMarker;
    let streetViewVisible = false;

    function initMap() {
        const defaultCenter = { lat: 39.8283, lng: -98.5795 }; // US center fallback
        const hasCoords = PROPERTY.lat !== null && PROPERTY.lng !== null;

        map = new google.maps.Map(document.getElementById('map'), {
            center: hasCoords ? { lat: PROPERTY.lat, lng: PROPERTY.lng } : defaultCenter,
            zoom: hasCoords ? 15 : 4,
            mapTypeId: 'roadmap',
            streetViewControl: false,
            fullscreenControl: true,
        });

        if (hasCoords) {
            placePropertyMarker(PROPERTY.lat, PROPERTY.lng);
        } else if (PROPERTY.address) {
            geocodeAddress(PROPERTY.address);
        }

        wireMapControls();
    }

    function placePropertyMarker(lat, lng) {
        const position = { lat, lng };
        if (!map) return;

        if (propertyMarker) {
            propertyMarker.setMap(null);
        }
        propertyMarker = new google.maps.Marker({
            map,
            position,
            title: PROPERTY.address || 'Subject Property',
        });

        map.setCenter(position);
        map.setZoom(15);

        const info = new google.maps.InfoWindow({
            content: `
                <div style="font-size:12px; max-width: 220px;">
                    <strong>Subject Property</strong><br>
                    ${PROPERTY.address ? PROPERTY.address.replace(/</g, '&lt;') : 'No address'}
                    <br><span style="color:#6b7280;">${lat.toFixed(5)}, ${lng.toFixed(5)}</span>
                </div>
            `
        });
        propertyMarker.addListener('click', () => info.open(map, propertyMarker));
    }

    function geocodeAddress(address) {
        const geocoder = new google.maps.Geocoder();
        geocoder.geocode({ address }, (results, status) => {
            if (status === 'OK' && results[0]) {
                const loc = results[0].geometry.location;
                placePropertyMarker(loc.lat(), loc.lng());
            } else {
                console.warn('Geocode failed:', status);
            }
        });
    }

    function wireMapControls() {
        const buttons = document.querySelectorAll('.map-controls .btn[data-map-type]');
        buttons.forEach(btn => {
            btn.addEventListener('click', () => {
                const type = btn.getAttribute('data-map-type');
                map.setMapTypeId(type);

                buttons.forEach(b => b.classList.remove('active'));
                btn.classList.add('active');
            });
        });

        const centerBtn = document.getElementById('btn-center-property');
        if (centerBtn) {
            centerBtn.addEventListener('click', () => {
                if (propertyMarker) {
                    map.panTo(propertyMarker.getPosition());
                    map.setZoom(15);
                } else if (PROPERTY.address) {
                    geocodeAddress(PROPERTY.address);
                }
            });
        }

        const streetViewBtn = document.getElementById('btn-toggle-streetview');
        if (streetViewBtn) {
            streetViewBtn.addEventListener('click', () => {
                toggleStreetView();
            });
        }
    }

    function toggleStreetView() {
        if (!map) return;

        const streetView = map.getStreetView();
        const isVisible = streetView.getVisible();

        if (!isVisible) {
            // Enter Street View at property marker if we have it; otherwise center
            let position;
            if (propertyMarker) {
                position = propertyMarker.getPosition();
            } else if (PROPERTY.lat && PROPERTY.lng) {
                position = { lat: PROPERTY.lat, lng: PROPERTY.lng };
            } else {
                // If no coords, try to geocode address one more time then toggle
                if (PROPERTY.address) {
                    geocodeAddress(PROPERTY.address);
                }
                return;
            }
            streetView.setPosition(position);
            streetView.setPov({ heading: 0, pitch: 0 });
            streetView.setVisible(true);
        } else {
            streetView.setVisible(false);
        }

        streetViewVisible = !isVisible;
    }
</script>
<?php endif; ?>
</body>
</html>
