<?php
// includes/sales-comps.php

$configPath = __DIR__ . '/../api/config.php';
if (file_exists($configPath)) {
    require_once $configPath;
}

$pdo = $pdo ?? null;
if (!$pdo && defined('DB_HOST') && defined('DB_NAME') && defined('DB_USER') && defined('DB_PASS')) {
    try {
        $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4';
        $pdo = new PDO($dsn, DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
    } catch (Exception $e) {
        $pdo = null;
    }
}

$errors = [];
$property = null;
$comps = [];

$propertyId = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$pdo) {
    $errors[] = 'Database connection not available.';
} elseif ($propertyId > 0) {
    try {
        // Base property
        $stmt = $pdo->prepare("
            SELECT id, address, city, state, zip, arv, purchase_price, rehab_budget, closing_costs
            FROM properties
            WHERE id = :id
            LIMIT 1
        ");
        $stmt->execute([':id' => $propertyId]);
        $property = $stmt->fetch();

        // Sales comps table (adjust table/field names as needed)
        $stmt = $pdo->prepare("
            SELECT
                id,
                subject_property_id,
                comp_address,
                comp_city,
                comp_state,
                comp_zip,
                distance_miles,
                beds,
                baths,
                sqft,
                year_built,
                sold_date,
                sold_price,
                list_price,
                days_on_market,
                source,
                created_at
            FROM sales_comps
            WHERE subject_property_id = :pid
            ORDER BY distance_miles ASC, sold_date DESC
        ");
        $stmt->execute([':pid' => $propertyId]);
        $comps = $stmt->fetchAll();
    } catch (Exception $e) {
        $errors[] = 'Error loading sales comps. Verify the sales_comps table.';
    }
} else {
    $errors[] = 'Missing property ID. Open this page from a specific property.';
}

// Quick stats
$avgPrice = $lowPrice = $highPrice = 0;
$avgPpsf = 0;
$countComps = count($comps);

if ($countComps > 0) {
    $prices = [];
    $ppsfList = [];

    foreach ($comps as $c) {
        if (!empty($c['sold_price'])) {
            $prices[] = (float)$c['sold_price'];
            if (!empty($c['sqft']) && $c['sqft'] > 0) {
                $ppsfList[] = (float)$c['sold_price'] / (float)$c['sqft'];
            }
        }
    }

    if (!empty($prices)) {
        sort($prices);
        $lowPrice  = $prices[0];
        $highPrice = $prices[count($prices) - 1];
        $avgPrice  = array_sum($prices) / count($prices);
    }

    if (!empty($ppsfList)) {
        $avgPpsf = array_sum($ppsfList) / count($ppsfList);
    }
}

function fmt_money($n) {
    if ($n <= 0) return '—';
    return '$' . number_format($n, 0);
}
function fmt_ppsf($n) {
    if ($n <= 0) return '—';
    return '$' . number_format($n, 0) . '/sf';
}
function fmt_pct($n) {
    return number_format($n, 1) . '%';
}
function fmt_date_s($d) {
    if (!$d) return '—';
    $ts = strtotime($d);
    if (!$ts) return $d;
    return date('M j, Y', $ts);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>MLS9 Deal Analyzer | Sales Comps</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        :root {
            --bg: #070b12;
            --bg-elevated: #0f172a;
            --accent: #22c55e;
            --border-subtle: rgba(148,163,184,0.25);
            --text-main: #e5e7eb;
            --text-muted: #9ca3af;
            --radius-xl: 26px;
            --shadow-soft: 0 18px 45px rgba(15,23,42,0.75);
        }
        *{box-sizing:border-box;}
        body{
            margin:0;
            font-family:system-ui,-apple-system,BlinkMacSystemFont,"SF Pro Text","Segoe UI",sans-serif;
            background:radial-gradient(circle at top left,#1e293b 0,#020617 40%,#000 100%);
            color:var(--text-main);
            min-height:100vh;
            display:flex;
        }
        a{text-decoration:none;color:inherit;}
        .layout{display:flex;width:100%;min-height:100vh;}
        .sidebar-shell{
            width:260px;
            background:#020617;
            border-right:1px solid rgba(15,23,42,0.9);
            box-shadow:12px 0 40px rgba(0,0,0,0.6);
            z-index:10;
        }
        .main{flex:1;display:flex;flex-direction:column;min-width:0;}
        .topbar-shell{
            position:sticky;top:0;z-index:9;
            backdrop-filter:blur(16px);
            background:linear-gradient(90deg,rgba(15,23,42,0.9),rgba(15,23,42,0.75),rgba(15,23,42,0.9));
            border-bottom:1px solid rgba(30,64,175,0.55);
        }
        .topbar-inner{
            max-width:1240px;margin:0 auto;padding:14px 22px;
            display:flex;align-items:center;justify-content:space-between;gap:16px;
        }
        .brand-lockup{display:flex;align-items:center;gap:10px;}
        .brand-mark{
            width:34px;height:34px;border-radius:999px;
            border:1px solid rgba(148,163,184,0.4);
            background:radial-gradient(circle at 20% 0,#22c55e 0,#0ea5e9 35%,#1d2435 100%);
            box-shadow:0 0 0 1px rgba(15,23,42,0.9),0 12px 30px rgba(34,197,94,0.35);
            display:flex;align-items:center;justify-content:center;font-weight:700;font-size:19px;
        }
        .brand-text{display:flex;flex-direction:column;gap:2px;}
        .brand-text span:first-child{
            font-size:14px;letter-spacing:0.18em;text-transform:uppercase;color:#9ca3af;
        }
        .brand-text span:last-child{font-size:18px;font-weight:600;}
        .pill{
            padding:4px 10px;border-radius:999px;
            border:1px solid rgba(148,163,184,0.4);
            background:radial-gradient(circle at top left,rgba(34,197,94,0.2),transparent 60%);
            display:inline-flex;align-items:center;gap:6px;
            font-size:13px;color:var(--text-muted);
        }
        .pill-dot{
            width:8px;height:8px;border-radius:999px;background:#22c55e;
            box-shadow:0 0 0 4px rgba(34,197,94,0.35);
        }
        .content-wrap{
            max-width:1240px;margin:22px auto 40px;
            padding:0 22px 22px;display:flex;flex-direction:column;gap:18px;
        }
        .page-header{display:flex;align-items:flex-start;justify-content:space-between;gap:14px;}
        .page-title{font-size:22px;font-weight:600;}
        .page-sub{font-size:13px;color:var(--text-muted);}
        .badge{
            padding:4px 9px;border-radius:999px;
            background:rgba(15,23,42,0.8);
            border:1px solid rgba(148,163,184,0.5);
            font-size:11px;text-transform:uppercase;letter-spacing:0.12em;color:var(--text-muted);
        }
        .card{
            border-radius:var(--radius-xl);
            background:radial-gradient(circle at top left,rgba(34,197,94,0.08),rgba(15,23,42,0.96));
            border:1px solid var(--border-subtle);
            box-shadow:var(--shadow-soft);
            padding:18px 18px 16px;position:relative;overflow:hidden;
        }
        .card-inner{position:relative;z-index:1;}
        .section-heading{
            font-size:13px;text-transform:uppercase;letter-spacing:0.14em;color:#9ca3af;margin-bottom:6px;
        }
        .card-subtitle{font-size:12px;color:var(--text-muted);margin-bottom:12px;}
        .grid-top{display:grid;grid-template-columns:1.3fr 1.2fr;gap:18px;}
        @media(max-width:960px){
            .sidebar-shell{display:none;}
            .grid-top{grid-template-columns:1fr;}
        }
        .summary-grid{
            display:grid;grid-template-columns:repeat(2,minmax(0,1fr));gap:10px;margin-top:8px;font-size:13px;
        }
        @media(max-width:720px){.summary-grid{grid-template-columns:1fr;}}
        .summary-item{
            border-radius:14px;padding:8px 10px;
            border:1px solid rgba(148,163,184,0.5);
            background:rgba(15,23,42,0.95);
        }
        .summary-label{
            font-size:11px;color:var(--text-muted);
            text-transform:uppercase;letter-spacing:0.12em;margin-bottom:2px;
        }
        .summary-value{font-size:15px;font-weight:500;}
        .summary-note{font-size:11px;color:var(--text-muted);margin-top:2px;}
        .good{color:#4ade80;}
        .warn{color:#facc15;}
        .bad{color:#f97373;}
        .btn-primary,.btn-secondary{
            border-radius:999px;display:inline-flex;align-items:center;gap:7px;
            font-size:13px;cursor:pointer;white-space:nowrap;
        }
        .btn-primary{
            padding:9px 16px;border:none;
            background:linear-gradient(135deg,#22c55e,#16a34a);color:#020617;
        }
        .btn-secondary{
            padding:8px 14px;border:1px solid rgba(148,163,184,0.6);
            background:rgba(15,23,42,0.9);color:var(--text-main);
        }
        .btn-primary span.icon,.btn-secondary span.icon{font-size:15px;}
        .messages{margin-top:6px;}
        .error-box{
            border-radius:12px;padding:8px 11px;font-size:12px;
            border:1px solid rgba(248,113,113,0.75);
            background:rgba(127,29,29,0.7);margin-bottom:6px;
        }
        .error-box ul{margin:0;padding-left:18px;}
        .error-box li{margin-bottom:3px;}
        .table-wrap{
            margin-top:12px;border-radius:18px;
            border:1px solid rgba(30,64,175,0.7);
            background:radial-gradient(circle at top left,rgba(30,64,175,0.32),rgba(15,23,42,0.98));
            overflow:hidden;
        }
        table{
            width:100%;border-collapse:collapse;font-size:12px;
        }
        thead{background:rgba(15,23,42,0.96);}
        th,td{padding:7px 9px;text-align:left;}
        th{
            font-size:11px;text-transform:uppercase;letter-spacing:0.12em;color:#a5b4fc;
            border-bottom:1px solid rgba(15,23,42,0.9);
        }
        tbody tr:nth-child(odd){background:rgba(15,23,42,0.9);}
        tbody tr:nth-child(even){background:rgba(15,23,42,0.82);}
        tbody tr:hover{background:rgba(30,64,175,0.6);}
        .pill-small{
            border-radius:999px;padding:3px 8px;
            border:1px solid rgba(148,163,184,0.6);
            font-size:11px;color:var(--text-muted);
        }
        .actions-row{display:flex;flex-wrap:wrap;gap:8px;margin-top:8px;}
        @media(max-width:540px){
            .page-header{flex-direction:column;align-items:flex-start;}
        }
    </style>
</head>
<body>
<div class="layout">
    <aside class="sidebar-shell">
        <?php
        $sidebarPath = __DIR__ . '/sidebar.php';
        if (file_exists($sidebarPath)) {
            include $sidebarPath;
        }
        ?>
    </aside>

    <div class="main">
        <header class="topbar-shell">
            <div class="topbar-inner">
                <div class="brand-lockup">
                    <div class="brand-mark">9</div>
                    <div class="brand-text">
                        <span>MLS9 | REI9</span>
                        <span>Sales comparables</span>
                    </div>
                </div>
                <div class="pill">
                    <span class="pill-dot"></span>
                    <span>Lock your ARV with real comps, not guesses</span>
                </div>
            </div>
        </header>

        <main class="content-wrap">
            <div class="page-header">
                <div>
                    <div class="page-title">
                        Sales comps
                        <?php if ($property): ?>
                            for “<?= htmlspecialchars($property['address']) ?>”
                        <?php endif; ?>
                    </div>
                    <div class="page-sub">
                        Pull recent solds around your subject property, sanity check ARV, and see your spread against the market.
                    </div>
                </div>
                <div>
                    <span class="badge">Step 3 — Sales comps</span>
                </div>
            </div>

            <?php if (!empty($errors)): ?>
                <div class="messages">
                    <div class="error-box">
                        <strong>Fix the following:</strong>
                        <ul>
                            <?php foreach ($errors as $e): ?>
                                <li><?= htmlspecialchars($e) ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                </div>
            <?php endif; ?>

            <section class="card">
                <div class="card-inner">
                    <div class="grid-top">
                        <div>
                            <div class="section-heading">Market snapshot</div>
                            <div class="card-subtitle">
                                Based on <?= $countComps ?> comps saved for this property.
                                Use this as your ARV guardrail before you lock in your offer.
                            </div>

                            <div class="summary-grid">
                                <div class="summary-item">
                                    <div class="summary-label">Average sold price</div>
                                    <div class="summary-value"><?= fmt_money($avgPrice) ?></div>
                                    <div class="summary-note">
                                        Across all saved comps.
                                    </div>
                                </div>
                                <div class="summary-item">
                                    <div class="summary-label">Low / high sale</div>
                                    <div class="summary-value">
                                        <?= fmt_money($lowPrice) ?> &mdash; <?= fmt_money($highPrice) ?>
                                    </div>
                                    <div class="summary-note">
                                        Range of comp sale prices.
                                    </div>
                                </div>
                                <div class="summary-item">
                                    <div class="summary-label">Avg price per SF</div>
                                    <div class="summary-value"><?= fmt_ppsf($avgPpsf) ?></div>
                                    <div class="summary-note">
                                        Use this to sanity check your ARV vs subject SQFT.
                                    </div>
                                </div>
                                <div class="summary-item">
                                    <div class="summary-label">Count of comps</div>
                                    <div class="summary-value"><?= (int)$countComps ?></div>
                                    <div class="summary-note">
                                        Aim for at least 5–8 strong comps.
                                    </div>
                                </div>
                            </div>

                            <div class="actions-row">
                                <?php if ($propertyId): ?>
                                    <button type="button" class="btn-primary" id="btn-run-sales-comps">
                                        <span class="icon">⚡</span>
                                        Run / refresh sales comps
                                    </button>
                                    <span class="pill-small">
                                        Calls <code>../api/sales-comps.php?property_id=<?= $propertyId ?></code> to update DB.
                                    </span>
                                <?php else: ?>
                                    <span class="pill-small">
                                        Open this page from a specific property to run comps.
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>

                        <div>
                            <div class="section-heading">Subject vs market</div>
                            <div class="card-subtitle">
                                Quick comparison of your current ARV and total project cost vs average comp price.
                            </div>
                            <?php
                            $subjectArv = $property['arv'] ?? 0;
                            $purchase   = $property['purchase_price'] ?? 0;
                            $rehab      = $property['rehab_budget'] ?? 0;
                            $closing    = $property['closing_costs'] ?? 0;
                            $totalCost  = $purchase + $rehab + $closing;
                            $subjectPct = ($subjectArv > 0 && $totalCost > 0) ? ($totalCost / $subjectArv * 100) : 0;
                            $avgVsSubjectPct = ($subjectArv > 0 && $avgPrice > 0) ? ($avgPrice / $subjectArv * 100) : 0;
                            ?>
                            <div class="summary-grid">
                                <div class="summary-item">
                                    <div class="summary-label">Subject ARV</div>
                                    <div class="summary-value"><?= fmt_money($subjectArv) ?></div>
                                    <div class="summary-note">
                                        From your purchase / analysis page.
                                    </div>
                                </div>
                                <div class="summary-item">
                                    <div class="summary-label">Total project cost</div>
                                    <div class="summary-value"><?= fmt_money($totalCost) ?></div>
                                    <div class="summary-note">
                                        Purchase + rehab + closing.
                                    </div>
                                </div>
                                <div class="summary-item">
                                    <div class="summary-label">Cost / ARV %</div>
                                    <div class="summary-value <?= $subjectPct <= 70 ? 'good' : ($subjectPct <= 80 ? 'warn' : 'bad') ?>">
                                        <?= $subjectArv > 0 && $totalCost > 0 ? fmt_pct($subjectPct) : '—' ?>
                                    </div>
                                    <div class="summary-note">
                                        Under 70% is strong, under 80% is tight but workable.
                                    </div>
                                </div>
                                <div class="summary-item">
                                    <div class="summary-label">Avg comp / ARV %</div>
                                    <div class="summary-value">
                                        <?= ($subjectArv > 0 && $avgPrice > 0) ? fmt_pct($avgVsSubjectPct) : '—' ?>
                                    </div>
                                    <div class="summary-note">
                                        How the average comp compares to your ARV.
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div style="margin-top:16px;" class="section-heading">Saved comps</div>
                    <div class="card-subtitle">
                        These comps are stored in your database. You can edit them later through your admin or API flow.
                    </div>

                    <div class="table-wrap">
                        <table>
                            <thead>
                            <tr>
                                <th>Address</th>
                                <th>Dist (mi)</th>
                                <th>Beds</th>
                                <th>Baths</th>
                                <th>Sqft</th>
                                <th>Sold price</th>
                                <th>Sold date</th>
                                <th>PPSF</th>
                                <th>DOM</th>
                                <th>Source</th>
                            </tr>
                            </thead>
                            <tbody id="sales-comps-body">
                            <?php if ($countComps === 0): ?>
                                <tr>
                                    <td colspan="10">No sales comps saved yet. Click “Run / refresh sales comps” to pull them from your APIs.</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($comps as $c): ?>
                                    <?php
                                    $ppsft = (!empty($c['sold_price']) && !empty($c['sqft']) && $c['sqft'] > 0)
                                        ? $c['sold_price'] / $c['sqft']
                                        : 0;
                                    ?>
                                    <tr>
                                        <td>
                                            <?= htmlspecialchars($c['comp_address']) ?><br>
                                            <span style="color:var(--text-muted);font-size:11px;">
                                                <?= htmlspecialchars($c['comp_city'] . ', ' . $c['comp_state'] . ' ' . $c['comp_zip']) ?>
                                            </span>
                                        </td>
                                        <td><?= $c['distance_miles'] !== null ? number_format($c['distance_miles'], 2) : '—' ?></td>
                                        <td><?= $c['beds'] !== null ? (float)$c['beds'] : '—' ?></td>
                                        <td><?= $c['baths'] !== null ? (float)$c['baths'] : '—' ?></td>
                                        <td><?= $c['sqft'] ? number_format($c['sqft']) : '—' ?></td>
                                        <td><?= $c['sold_price'] ? fmt_money($c['sold_price']) : '—' ?></td>
                                        <td><?= fmt_date_s($c['sold_date']) ?></td>
                                        <td><?= $ppsft > 0 ? '$' . number_format($ppsft, 0) : '—' ?></td>
                                        <td><?= $c['days_on_market'] !== null ? (int)$c['days_on_market'] : '—' ?></td>
                                        <td><?= htmlspecialchars($c['source'] ?: 'API') ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                            </tbody>
                        </table>
                    </div>

                    <div style="margin-top:8px;font-size:11px;color:var(--text-muted);">
                        Sales comps flow: Property → Pull from APIs → Save to <code>sales_comps</code> table → Use here, Analysis, and Reports.
                    </div>
                </div>
            </section>
        </main>
    </div>
</div>

<script>
const pid = <?= json_encode($propertyId) ?>;
const btnSales = document.getElementById('btn-run-sales-comps');

if (btnSales && pid) {
    btnSales.addEventListener('click', () => {
        btnSales.disabled = true;
        btnSales.innerHTML = '<span class="icon">⏳</span>Running comps...';
        fetch('../api/sales-comps.php?property_id=' + encodeURIComponent(pid), {
            method: 'GET',
            headers: { 'Accept': 'application/json' }
        })
            .then(r => r.json().catch(() => ({})))
            .then(data => {
                btnSales.disabled = false;
                btnSales.innerHTML = '<span class="icon">⚡</span>Run / refresh sales comps';
                if (data && data.success) {
                    location.reload();
                } else {
                    alert('Sales comps API did not return success. Check ../api/sales-comps.php.');
                }
            })
            .catch(() => {
                btnSales.disabled = false;
                btnSales.innerHTML = '<span class="icon">⚡</span>Run / refresh sales comps';
                alert('Error calling sales comps API.');
            });
    });
}
</script>
</body>
</html>
