// Front page logic: search + local list rendering

document.addEventListener('DOMContentLoaded', function () {
  const searchForm = document.getElementById('search-form');
  const addressInput = document.getElementById('address-input');
  const suggestionsBox = document.getElementById('address-suggestions');
  const statusEl = document.getElementById('search-status');
  const listContainer = document.getElementById('properties-list');
  const emptyEl = document.getElementById('properties-list-empty');

  // Render existing properties from localStorage
  renderLocalProperties();

  function renderLocalProperties() {
    const list = loadLocalProperties();
    listContainer.innerHTML = '';
    if (!list || list.length === 0) {
      emptyEl.style.display = 'block';
      return;
    }
    emptyEl.style.display = 'none';
    list.forEach(p => {
      const div = document.createElement('div');
      div.className = 'panel';
      div.innerHTML = `
        <div class="panel-body">
          <div class="summary-item-block">
            <div class="summary-line-item">
              <a href="property.html?id=${encodeURIComponent(p.id)}" class="blue bold">
                ${p.address_line1 || p.formatted_address || 'Property'}
              </a>
            </div>
            <div class="summary-line-item">
              ${p.city || ''}, ${p.state || ''} ${p.zip || ''}
            </div>
            <div class="summary-line-item">
              ${p.beds || '-'} bd &bull; ${p.baths || '-'} ba &bull; ${p.sqft ? p.sqft + ' sf' : '-'}
            </div>
            <div class="summary-line-item-total">
              Est. Value: ${formatCurrency(p.est_value)} |
              Est. Rent: ${formatCurrency(p.est_rent)}
            </div>
          </div>
        </div>`;
      listContainer.appendChild(div);
    });
  }

  // Address suggestions (ChatGPT-backed via backend)
  let suggestionTimeout = null;
  addressInput.addEventListener('input', function () {
    const q = addressInput.value.trim();
    if (!q || q.length < 4) {
      suggestionsBox.innerHTML = '';
      return;
    }
    if (suggestionTimeout) clearTimeout(suggestionTimeout);
    suggestionTimeout = setTimeout(() => fetchSuggestions(q), 400);
  });

  function fetchSuggestions(query) {
    fetch('api/address_suggest.php?q=' + encodeURIComponent(query))
      .then(r => r.json())
      .then(data => {
        suggestionsBox.innerHTML = '';
        if (!data || !Array.isArray(data.suggestions) || data.suggestions.length === 0) return;
        const ul = document.createElement('ul');
        data.suggestions.forEach(s => {
          const li = document.createElement('li');
          li.textContent = s;
          li.addEventListener('click', () => {
            addressInput.value = s;
            suggestionsBox.innerHTML = '';
          });
          ul.appendChild(li);
        });
        suggestionsBox.appendChild(ul);
      })
      .catch(err => {
        console.warn('Suggestion error', err);
      });
  }

  searchForm.addEventListener('submit', function (e) {
    e.preventDefault();
    suggestionsBox.innerHTML = '';
    const address = addressInput.value.trim();
    const city = document.getElementById('city-input').value.trim();
    const state = document.getElementById('state-input').value.trim();
    const zip = document.getElementById('zip-input').value.trim();

    if (!address) {
      statusEl.textContent = 'Please enter an address.';
      return;
    }

    statusEl.textContent = 'Analyzing property...';

    const formData = new FormData();
    formData.append('action', 'get_or_create');
    formData.append('address', address);
    formData.append('city', city);
    formData.append('state', state);
    formData.append('zip', zip);

    fetch('api/property.php', {
      method: 'POST',
      body: formData
    })
      .then(r => r.json())
      .then(data => {
        if (data.error) {
          statusEl.textContent = 'Error: ' + data.error;
          return;
        }
        statusEl.textContent = '';
        const property = data.property;
        savePropertyToLocalList(property);
        renderLocalProperties();
        window.location.href = 'property.html?id=' + encodeURIComponent(property.id);
      })
      .catch(err => {
        console.error(err);
        statusEl.textContent = 'There was an error analyzing this property.';
      });
  });
});
