<?php
// api/property.php
header('Content-Type: application/json');

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/rentcast_client.php';

$config = require __DIR__ . '/config.php';

$action = $_POST['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'get_or_create':
        mls9_get_or_create_property();
        break;
    case 'get':
        mls9_get_property();
        break;
    case 'delete':
        mls9_delete_property();
        break;
    default:
        echo json_encode(['error' => 'Invalid action']);
        break;
}

function mls9_normalize_address_key($address, $city, $state, $zip) {
    $parts = array_filter([$address, $city, $state, $zip]);
    $norm = strtolower(trim(implode(', ', $parts)));
    $norm = preg_replace('/\s+/', ' ', $norm);
    return $norm;
}

function mls9_get_or_create_property() {
    global $config;
    $pdo = mls9_get_pdo();

    $address = trim($_POST['address'] ?? '');
    $city = trim($_POST['city'] ?? '');
    $state = trim($_POST['state'] ?? '');
    $zip = trim($_POST['zip'] ?? '');

    if ($address === '') {
        echo json_encode(['error' => 'Address is required']);
        return;
    }

    $normalized = mls9_normalize_address_key($address, $city, $state, $zip);

    // Check DB cache
    $stmt = $pdo->prepare('SELECT * FROM properties WHERE normalized_address = :norm LIMIT 1');
    $stmt->execute([':norm' => $normalized]);
    $row = $stmt->fetch();

    $now = new DateTime('now', new DateTimeZone('UTC'));
    $needsRefresh = false;
    if ($row) {
        $updated = new DateTime($row['updated_at'], new DateTimeZone('UTC'));
        $days = ($now->getTimestamp() - $updated->getTimestamp()) / 86400;
        if ($days > $config['cache_ttl_days']) {
            $needsRefresh = true;
        }
    }

    if ($row && !$needsRefresh) {
        $property = mls9_row_to_property($row);
        echo json_encode(['property' => $property]);
        return;
    }

    // Fetch from RentCast
    $bundle = mls9_fetch_rentcast_bundle($address, $city, $state, $zip);
    if (!$bundle || (!$bundle['property'] && !$bundle['value'] && !$bundle['rent'])) {
        if ($row) {
            $property = mls9_row_to_property($row);
            echo json_encode(['property' => $property, 'warning' => 'RentCast data could not be refreshed. Showing cached data.']);
        } else {
            echo json_encode(['error' => 'Unable to fetch property data from RentCast']);
        }
        return;
    }

    $propertyCore = mls9_bundle_to_property($bundle, $address, $city, $state, $zip, $normalized);

    if ($row) {
        $stmt = $pdo->prepare('UPDATE properties
            SET address_line1 = :addr1, city = :city, state = :state, zip = :zip,
                formatted_address = :faddr, beds = :beds, baths = :baths,
                sqft = :sqft, property_type = :ptype,
                est_value = :est_value, est_rent = :est_rent,
                rentcast_property = :rc_property, rentcast_value = :rc_value,
                rentcast_rent = :rc_rent,
                worksheet = :worksheet, mls9_data = :mls9_data,
                updated_at = :updated_at
            WHERE id = :id');

        $stmt->execute([
            ':addr1' => $propertyCore['address_line1'],
            ':city' => $propertyCore['city'],
            ':state' => $propertyCore['state'],
            ':zip' => $propertyCore['zip'],
            ':faddr' => $propertyCore['formatted_address'],
            ':beds' => $propertyCore['beds'],
            ':baths' => $propertyCore['baths'],
            ':sqft' => $propertyCore['sqft'],
            ':ptype' => $propertyCore['property_type'],
            ':est_value' => $propertyCore['est_value'],
            ':est_rent' => $propertyCore['est_rent'],
            ':rc_property' => json_encode($bundle['property']),
            ':rc_value' => json_encode($bundle['value']),
            ':rc_rent' => json_encode($bundle['rent']),
            ':worksheet' => json_encode($propertyCore['worksheet']),
            ':mls9_data' => json_encode($propertyCore),
            ':updated_at' => $now->format('Y-m-d H:i:s'),
            ':id' => $row['id'],
        ]);
        $id = $row['id'];
    } else {
        $stmt = $pdo->prepare('INSERT INTO properties
            (normalized_address, address_line1, city, state, zip, formatted_address,
             beds, baths, sqft, property_type, est_value, est_rent,
             rentcast_property, rentcast_value, rentcast_rent,
             worksheet, mls9_data, created_at, updated_at)
            VALUES
            (:norm, :addr1, :city, :state, :zip, :faddr,
             :beds, :baths, :sqft, :ptype, :est_value, :est_rent,
             :rc_property, :rc_value, :rc_rent,
             :worksheet, :mls9_data, :created_at, :updated_at)');
        $stmt->execute([
            ':norm' => $normalized,
            ':addr1' => $propertyCore['address_line1'],
            ':city' => $propertyCore['city'],
            ':state' => $propertyCore['state'],
            ':zip' => $propertyCore['zip'],
            ':faddr' => $propertyCore['formatted_address'],
            ':beds' => $propertyCore['beds'],
            ':baths' => $propertyCore['baths'],
            ':sqft' => $propertyCore['sqft'],
            ':ptype' => $propertyCore['property_type'],
            ':est_value' => $propertyCore['est_value'],
            ':est_rent' => $propertyCore['est_rent'],
            ':rc_property' => json_encode($bundle['property']),
            ':rc_value' => json_encode($bundle['value']),
            ':rc_rent' => json_encode($bundle['rent']),
            ':worksheet' => json_encode($propertyCore['worksheet']),
            ':mls9_data' => json_encode($propertyCore),
            ':created_at' => $now->format('Y-m-d H:i:s'),
            ':updated_at' => $now->format('Y-m-d H:i:s'),
        ]);
        $id = $pdo->lastInsertId();
    }

    $stmt = $pdo->prepare('SELECT * FROM properties WHERE id = :id');
    $stmt->execute([':id' => $id]);
    $rowNew = $stmt->fetch();
    $property = mls9_row_to_property($rowNew);
    echo json_encode(['property' => $property]);
}

function mls9_get_property() {
    $id = intval($_POST['id'] ?? $_GET['id'] ?? 0);
    if ($id <= 0) {
        echo json_encode(['error' => 'Invalid id']);
        return;
    }
    $pdo = mls9_get_pdo();
    $stmt = $pdo->prepare('SELECT * FROM properties WHERE id = :id');
    $stmt->execute([':id' => $id]);
    $row = $stmt->fetch();
    if (!$row) {
        echo json_encode(['error' => 'Property not found']);
        return;
    }
    $property = mls9_row_to_property($row);
    echo json_encode(['property' => $property]);
}

function mls9_delete_property() {
    $id = intval($_POST['id'] ?? 0);
    if ($id <= 0) {
        echo json_encode(['error' => 'Invalid id']);
        return;
    }
    $pdo = mls9_get_pdo();
    $stmt = $pdo->prepare('DELETE FROM properties WHERE id = :id');
    $stmt->execute([':id' => $id]);
    echo json_encode(['success' => true]);
}

function mls9_row_to_property(array $row) {
    $p = json_decode($row['mls9_data'], true);
    if (!is_array($p)) $p = [];
    $p['id'] = (int)$row['id'];
    $p['rentcast_property'] = json_decode($row['rentcast_property'], true) ?: [];
    $p['rentcast_value'] = json_decode($row['rentcast_value'], true) ?: [];
    $p['rentcast_rent'] = json_decode($row['rentcast_rent'], true) ?: [];
    return $p;
}

function mls9_bundle_to_property(array $bundle, $address, $city, $state, $zip, $normalized) {
    $prop = null;
    if (is_array($bundle['property'])) {
        // RentCast may return either an array of properties or a single object
        if (isset($bundle['property'][0])) {
            $prop = $bundle['property'][0];
        } else {
            $prop = $bundle['property'];
        }
    } else {
        $prop = [];
    }
    $value = is_array($bundle['value']) ? $bundle['value'] : [];
    $rent = is_array($bundle['rent']) ? $bundle['rent'] : [];

    $formattedAddress = $prop['formattedAddress'] ?? $prop['address'] ?? $address;

    $beds = $prop['bedrooms'] ?? $prop['beds'] ?? null;
    $baths = $prop['bathrooms'] ?? $prop['baths'] ?? null;
    $sqft = $prop['sqft'] ?? $prop['squareFootage'] ?? null;
    $propertyType = $prop['propertyType'] ?? $prop['type'] ?? null;

    $estValue = $value['value'] ?? $value['estimate'] ?? null;
    $estRent = $rent['rent'] ?? $rent['estimate'] ?? null;

    $worksheet = [
        'purchase_price' => $estValue,
        'down_payment' => $estValue ? round($estValue * 0.2) : null,
        'interest_rate' => 7,
        'term_years' => 30,
        'closing_costs' => $estValue ? round($estValue * 0.03) : null,
        'rehab_budget' => 0,
    ];

    return [
        'normalized_address' => $normalized,
        'address_line1' => $prop['addressLine1'] ?? $address,
        'city' => $prop['city'] ?? $city,
        'state' => $prop['state'] ?? $state,
        'zip' => $prop['zip'] ?? $zip,
        'formatted_address' => $formattedAddress,
        'beds' => $beds,
        'baths' => $baths,
        'sqft' => $sqft,
        'property_type' => $propertyType,
        'est_value' => $estValue,
        'est_rent' => $estRent,
        'worksheet' => $worksheet,
        // placeholders for any additional MLS9-calculated data
        'annual_taxes' => $prop['taxes'] ?? null,
        'annual_insurance' => null,
        'annual_maintenance' => null,
        'vacancy_rate' => 5,
        'photos' => $prop['photos'] ?? [],
        'cap_rate' => null,
    ];
}
