<?php
// search.php
// Results page showing listings matching a search query. When no query is
// provided, all listings are displayed. Pagination is implemented for
// demonstration but with a small page size. The filter bar is static and
// illustrative; functional filters could be wired up via JavaScript or
// additional query parameters.

require_once __DIR__ . '/data.php';
include __DIR__ . '/header.php';

// Retrieve the search query and trim whitespace
$query = isset($_GET['query']) ? trim($_GET['query']) : '';

// Filter listings based on the query (match address, city, state or zip)
$filtered = [];
foreach ($listings as $listing) {
    $searchable = strtolower($listing['address'] . ' ' . $listing['city'] . ' ' . $listing['state'] . ' ' . $listing['zip']);
    if ($query === '' || strpos($searchable, strtolower($query)) !== false) {
        $filtered[] = $listing;
    }
}

// Pagination settings
$pageSize = 12;
$totalItems = count($filtered);
$totalPages = max(1, (int)ceil($totalItems / $pageSize));
$page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
if ($page < 1) $page = 1;
if ($page > $totalPages) $page = $totalPages;

$startIndex = ($page - 1) * $pageSize;
$pagedListings = array_slice($filtered, $startIndex, $pageSize);

?>

<div class="search-page">
    <div class="search-header">
        <h1>Homes<?php if ($query !== '') { echo ' near "' . htmlspecialchars($query) . '"'; } ?></h1>
        <p><?php echo $totalItems; ?> home<?php echo $totalItems === 1 ? '' : 's'; ?> found</p>
    </div>

    <div class="filterbar">
        <button class="filter-chip filter-chip-outline" type="button">For Sale</button>
        <button class="filter-chip" type="button">For Rent</button>
        <button class="filter-chip" type="button">Price</button>
        <button class="filter-chip" type="button">Beds &amp; Baths</button>
        <button class="filter-chip" type="button">Home Type</button>
        <button class="filter-chip" type="button">More</button>
    </div>

    <div class="search-layout">
        <div class="search-results-list">
            <?php if (empty($pagedListings)): ?>
                <div class="empty-state">
                    <h3>No homes match your search</h3>
                    <p>Try adjusting your filters or searching for a different city, neighborhood, or ZIP code.</p>
                </div>
            <?php else: ?>
                <div class="listing-grid">
                    <?php foreach ($pagedListings as $listing): ?>
                        <a href="/zillow/z0.0/listing.php?id=<?php echo urlencode($listing['id']); ?>" class="listing-card">
                            <div class="listing-card-image-wrap">
                                <img src="<?php echo htmlspecialchars($listing['imageUrl']); ?>" alt="<?php echo htmlspecialchars($listing['address']); ?>" class="listing-card-image">
                                <?php if (!empty($listing['badge'])): ?>
                                    <span class="badge"><?php echo htmlspecialchars($listing['badge']); ?></span>
                                <?php endif; ?>
                                <?php if ($listing['isSaved']): ?>
                                    <span class="listing-heart">♥</span>
                                <?php endif; ?>
                            </div>
                            <div class="listing-card-body">
                                <div class="listing-price">$<?php echo number_format($listing['price']); ?></div>
                                <div class="listing-meta">
                                    <span><?php echo (int)$listing['beds']; ?> bd</span> ·
                                    <span><?php echo (int)$listing['baths']; ?> ba</span> ·
                                    <span><?php echo number_format($listing['sqft']); ?> sqft</span>
                                </div>
                                <div class="listing-address"><?php echo htmlspecialchars($listing['address']); ?></div>
                                <div class="listing-subtext">
                                    <?php echo htmlspecialchars($listing['city'] . ', ' . $listing['state'] . ' ' . $listing['zip']); ?>
                                </div>
                            </div>
                        </a>
                    <?php endforeach; ?>
                </div>
                <?php if ($totalPages > 1): ?>
                    <div class="pagination">
                        <?php $prevPage = $page - 1; ?>
                        <?php $nextPage = $page + 1; ?>
                        <button type="button" <?php echo $page <= 1 ? 'disabled' : ''; ?> onclick="window.location.href='?<?php echo http_build_query(['query' => $query, 'page' => max(1, $prevPage)]); ?>'">Previous</button>
                        <span class="pagination-info">Page <?php echo $page; ?> of <?php echo $totalPages; ?></span>
                        <button type="button" <?php echo $page >= $totalPages ? 'disabled' : ''; ?> onclick="window.location.href='?<?php echo http_build_query(['query' => $query, 'page' => min($totalPages, $nextPage)]); ?>'">Next</button>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        <div class="map-pane">
            <div class="map-placeholder">
                <p>Map placeholder</p>
                <p>Embed your favourite map provider here (Leaflet, Google Maps, Mapbox, etc.).</p>
            </div>
        </div>
    </div>
</div>

<?php
include __DIR__ . '/footer.php';