<?php
// auth.php
require_once __DIR__ . '/db.php';

function create_session($user_id) {
    $pdo = get_pdo();

    // Generate secure random token
    $session_id = bin2hex(random_bytes(32));
    $expires_at = (new DateTime('now', new DateTimeZone('UTC')))
        ->add(new DateInterval('PT' . SESSION_LIFETIME . 'S'))
        ->format('Y-m-d H:i:s');

    $stmt = $pdo->prepare("
        INSERT INTO user_sessions (session_id, user_id, created_at, expires_at, ip_address, user_agent)
        VALUES (:sid, :uid, NOW(), :exp, :ip, :ua)
    ");
    $stmt->execute([
        ':sid' => $session_id,
        ':uid' => $user_id,
        ':exp' => $expires_at,
        ':ip'  => $_SERVER['REMOTE_ADDR'] ?? null,
        ':ua'  => $_SERVER['HTTP_USER_AGENT'] ?? null,
    ]);

    setcookie('session_id', $session_id, [
        'expires'  => time() + SESSION_LIFETIME,
        'path'     => '/',
        'secure'   => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
        'httponly' => true,
        'samesite' => 'Lax',
    ]);
}

function destroy_session() {
    if (!empty($_COOKIE['session_id'])) {
        $pdo = get_pdo();
        $stmt = $pdo->prepare("DELETE FROM user_sessions WHERE session_id = :sid");
        $stmt->execute([':sid' => $_COOKIE['session_id']]);

        setcookie('session_id', '', time() - 3600, '/');
    }
}

function get_current_user() {
    static $currentUser = null;
    if ($currentUser !== null) {
        return $currentUser;
    }

    if (empty($_COOKIE['session_id'])) {
        return null;
    }

    $pdo = get_pdo();
    $stmt = $pdo->prepare("
        SELECT u.*
        FROM user_sessions s
        JOIN users u ON u.user_id = s.user_id
        WHERE s.session_id = :sid
          AND s.expires_at > NOW()
          AND u.is_active = 1
        LIMIT 1
    ");
    $stmt->execute([':sid' => $_COOKIE['session_id']]);
    $user = $stmt->fetch();

    if (!$user) {
        // Clean up bad session cookie
        setcookie('session_id', '', time() - 3600, '/');
        return null;
    }

    $currentUser = $user;
    return $currentUser;
}

function require_login() {
    $user = get_current_user();
    if (!$user) {
        header('Location: login.php');
        exit;
    }
    return $user;
}
