// Property detail page logic

let currentProperty = null;

document.addEventListener('DOMContentLoaded', function () {
  const id = getQueryParam('id');
  if (!id) {
    document.getElementById('section-description').innerHTML =
      '<p class="help-block">No property selected.</p>';
    return;
  }

  loadProperty(id);
});

function loadProperty(id) {
  const params = new URLSearchParams();
  params.append('action', 'get');
  params.append('id', id);

  fetch('api/property.php', {
    method: 'POST',
    body: params
  })
    .then(r => r.json())
    .then(data => {
      if (data.error) {
        console.error(data.error);
        return;
      }
      currentProperty = data.property;
      populateSidebar(currentProperty);
      renderAllSections(currentProperty);
    })
    .catch(err => console.error(err));
}

function populateSidebar(p) {
  const addr = p.formatted_address || [p.address_line1, p.city, p.state, p.zip].filter(Boolean).join(', ');
  document.getElementById('sidebar-property-address').textContent = addr;
  document.getElementById('sidebar-property-location').textContent = `${p.city || ''}, ${p.state || ''} ${p.zip || ''}`;

  const meta = [];
  if (p.beds) meta.push(p.beds + ' bd');
  if (p.baths) meta.push(p.baths + ' ba');
  if (p.sqft) meta.push(p.sqft + ' sf');
  if (p.property_type) meta.push(p.property_type);
  document.getElementById('sidebar-property-meta').textContent = meta.join(' • ');

  document.getElementById('sidebar-price').textContent = formatCurrency(p.est_value);
  document.getElementById('sidebar-caprate').textContent = p.cap_rate ? formatPercent(p.cap_rate) : '';

  const mainPhoto = document.getElementById('property-main-photo');
  if (p.photos && p.photos.length > 0) {
    mainPhoto.src = p.photos[0];
  } else {
    mainPhoto.src = 'https://via.placeholder.com/600x360?text=MLS9+Property';
  }
}

function renderAllSections(p) {
  renderDescription(p);
  renderWorksheet(p);
  renderPhotos(p);
  renderMap(p);
  renderAnalysis(p);
  renderProjections(p);
  renderSalesComps(p);
  renderRentalComps(p);
  renderRecords(p);
  renderOwner(p);
  renderOfferCalculator(p);
  renderReports(p);
  renderCompare(p);
  renderLenders(p);
  renderTrack(p);
}

// SECTION RENDERERS

function renderDescription(p) {
  const el = document.getElementById('description-body');
  const rc = p.rentcast_property || {};
  const lines = [
    `<strong>${p.formatted_address || ''}</strong>`,
    `${p.city || ''}, ${p.state || ''} ${p.zip || ''}`,
    '',
    `<strong>Basic Info</strong>`,
    `Type: ${p.property_type || rc.type || '-'}`,
    `Beds: ${p.beds || rc.beds || '-'}`,
    `Baths: ${p.baths || rc.baths || '-'}`,
    `Square Feet: ${p.sqft || rc.sqft || '-'}`,
    `Lot Size: ${rc.lotSize ? rc.lotSize + ' sq ft' : '-'}`,
    `Year Built: ${rc.yearBuilt || '-'}`,
    `Stories: ${rc.stories || '-'}`,
    `Parking: ${rc.parking || '-'}`,
    '',
    `<strong>Location</strong>`,
    `County: ${rc.county || '-'}`,
    `Neighborhood: ${rc.neighborhood || '-'}`,
    `Latitude / Longitude: ${rc.lat || '-'} / ${rc.lng || '-'}`
  ];
  el.innerHTML = '<div class="panel"><div class="panel-body"><p>' +
    lines.join('<br/>') +
    '</p></div></div>';
}

function renderWorksheet(p) {
  const el = document.getElementById('worksheet-body');

  const defaults = p.worksheet || {};
  const purchasePrice = defaults.purchase_price || p.est_value || 0;
  const downPayment = defaults.down_payment || Math.round(purchasePrice * 0.2);
  const interestRate = defaults.interest_rate || 7;
  const termYears = defaults.term_years || 30;
  const closingCosts = defaults.closing_costs || Math.round(purchasePrice * 0.03);
  const rehabBudget = defaults.rehab_budget || 0;

  const html = `
    <div class="panel">
      <div class="panel-body">
        <div class="summary-item-block">
          <div class="summary-line-item">
            <label>Purchase Price</label>
            <input type="number" id="ws-purchase-price" value="${purchasePrice}" />
          </div>
          <div class="summary-line-item">
            <label>Down Payment ($)</label>
            <input type="number" id="ws-down-payment" value="${downPayment}" />
          </div>
          <div class="summary-line-item">
            <label>Interest Rate (%)</label>
            <input type="number" id="ws-interest-rate" value="${interestRate}" step="0.01" />
          </div>
          <div class="summary-line-item">
            <label>Amortization Term (years)</label>
            <input type="number" id="ws-term-years" value="${termYears}" />
          </div>
          <div class="summary-line-item">
            <label>Closing Costs ($)</label>
            <input type="number" id="ws-closing-costs" value="${closingCosts}" />
          </div>
          <div class="summary-line-item">
            <label>Rehab Budget ($)</label>
            <input type="number" id="ws-rehab-budget" value="${rehabBudget}" />
          </div>
          <div class="summary-line-item-total">
            <div>Loan Amount: <span id="ws-loan-amount"></span></div>
            <div>Monthly Payment (P&amp;I): <span id="ws-monthly-payment"></span></div>
          </div>
        </div>
        <div class="margin-t-10">
          <button class="btn btn-primary" onclick="recalculateWorksheet()">Recalculate</button>
        </div>
      </div>
    </div>
  `;
  el.innerHTML = html;
  recalculateWorksheet();
}

function recalculateWorksheet() {
  const purchasePrice = Number(document.getElementById('ws-purchase-price').value) || 0;
  const downPayment = Number(document.getElementById('ws-down-payment').value) || 0;
  const interestRate = Number(document.getElementById('ws-interest-rate').value) || 0;
  const termYears = Number(document.getElementById('ws-term-years').value) || 0;

  const loanAmount = purchasePrice - downPayment;
  const monthlyRate = interestRate / 100 / 12;
  const n = termYears * 12;
  let monthlyPayment = 0;
  if (monthlyRate > 0 && n > 0) {
    monthlyPayment = (loanAmount * monthlyRate * Math.pow(1 + monthlyRate, n)) /
      (Math.pow(1 + monthlyRate, n) - 1);
  }

  document.getElementById('ws-loan-amount').textContent = formatCurrency(loanAmount);
  document.getElementById('ws-monthly-payment').textContent = formatCurrency(monthlyPayment);
}

function renderPhotos(p) {
  const el = document.getElementById('photos-body');
  const photos = p.photos && p.photos.length ? p.photos : [];
  if (photos.length === 0) {
    el.innerHTML = '<p class="help-block">No photos available.</p>';
    return;
  }
  el.innerHTML = '';
  photos.forEach(src => {
    const img = document.createElement('img');
    img.src = src;
    el.appendChild(img);
  });
}

function renderMap(p) {
  const iframe = document.getElementById('map-iframe');
  const addr = encodeURIComponent(p.formatted_address || [p.address_line1, p.city, p.state, p.zip].filter(Boolean).join(', '));
  iframe.src = 'https://www.google.com/maps?q=' + addr + '&output=embed';
}

function renderAnalysis(p) {
  const el = document.getElementById('analysis-body');
  const value = p.est_value || 0;
  const rent = p.est_rent || 0;
  const taxes = p.annual_taxes || (value * 0.015);
  const insurance = p.annual_insurance || 1200;
  const maintenance = p.annual_maintenance || (rent * 0.08 * 12);
  const vacancyRate = p.vacancy_rate || 5;

  const grossAnnualRent = rent * 12;
  const vacancyLoss = grossAnnualRent * (vacancyRate / 100);
  const effectiveGross = grossAnnualRent - vacancyLoss;
  const operatingExpenses = taxes + insurance + maintenance;
  const noi = effectiveGross - operatingExpenses;

  const totalInvestment = (p.worksheet && p.worksheet.down_payment || value * 0.2) +
    (p.worksheet && p.worksheet.closing_costs || value * 0.03) +
    (p.worksheet && p.worksheet.rehab_budget || 0);

  const capRate = value ? (noi / value) * 100 : 0;
  const cocReturn = totalInvestment ? (noi / totalInvestment) * 100 : 0;

  p.cap_rate = capRate;

  const html = `
    <div class="panel">
      <div class="panel-body">
        <div class="summary-item-block">
          <div class="summary-line-item">Estimated Value: <span class="blue">${formatCurrency(value)}</span></div>
          <div class="summary-line-item">Estimated Rent: <span class="blue">${formatCurrency(rent)}</span></div>
          <div class="summary-line-item">Gross Annual Rent: ${formatCurrency(grossAnnualRent)}</div>
          <div class="summary-line-item">Vacancy Rate: ${formatPercent(vacancyRate)}</div>
          <div class="summary-line-item">Effective Gross Income: ${formatCurrency(effectiveGross)}</div>
          <div class="summary-line-item">Annual Taxes: ${formatCurrency(taxes)}</div>
          <div class="summary-line-item">Annual Insurance: ${formatCurrency(insurance)}</div>
          <div class="summary-line-item">Maintenance: ${formatCurrency(maintenance)}</div>
          <div class="summary-line-item">Operating Expenses: ${formatCurrency(operatingExpenses)}</div>
          <div class="summary-line-item-total">Net Operating Income (NOI): <span class="blue">${formatCurrency(noi)}</span></div>
          <div class="summary-line-item">Cap Rate: <span class="blue">${formatPercent(capRate)}</span></div>
          <div class="summary-line-item">Cash-on-Cash Return: <span class="blue">${formatPercent(cocReturn)}</span></div>
        </div>
      </div>
    </div>
    <div class="panel">
      <div class="panel-body">
        <div id="chart-expenses" style="height:260px;"></div>
      </div>
    </div>
  `;
  el.innerHTML = html;

  google.charts.load('current', { packages: ['corechart'] });
  google.charts.setOnLoadCallback(function () {
    const data = google.visualization.arrayToDataTable([
      ['Type', 'Amount'],
      ['Taxes', taxes],
      ['Insurance', insurance],
      ['Maintenance', maintenance],
      ['Vacancy Loss', vacancyLoss]
    ]);
    const options = {
      legend: { position: 'right' },
      pieHole: 0.4,
      chartArea: { width: '80%', height: '80%' }
    };
    const chart = new google.visualization.PieChart(document.getElementById('chart-expenses'));
    chart.draw(data, options);
  });
}

function renderProjections(p) {
  const el = document.getElementById('projections-body');
  const years = 10;
  const rent = p.est_rent || 0;
  const rentGrowth = 2;
  const value = p.est_value || 0;
  const valueGrowth = 3;

  let rows = '';
  let curRent = rent;
  let curValue = value;
  for (let y = 1; y <= years; y++) {
    curRent *= (1 + rentGrowth / 100);
    curValue *= (1 + valueGrowth / 100);
    rows += `<tr>
      <td>Year ${y}</td>
      <td>${formatCurrency(curRent * 12)}</td>
      <td>${formatCurrency(curValue)}</td>
    </tr>`;
  }

  el.innerHTML = `
    <div class="panel">
      <div class="panel-body">
        <table class="table">
          <thead>
            <tr>
              <th>Year</th>
              <th>Projected Annual Rent</th>
              <th>Projected Value</th>
            </tr>
          </thead>
          <tbody>${rows}</tbody>
        </table>
      </div>
    </div>
  `;
}

function renderSalesComps(p) {
  const el = document.getElementById('sales-comps-body');
  const comps = (p.rentcast_value && p.rentcast_value.salesComps) || [];
  if (!comps.length) {
    el.innerHTML = '<p class="help-block">No sales comparables available from RentCast.</p>';
    return;
  }
  let rows = '';
  comps.forEach(c => {
    rows += `<tr>
      <td>${c.address || ''}</td>
      <td>${c.distance ? c.distance.toFixed(2) + ' mi' : '-'}</td>
      <td>${formatCurrency(c.price)}</td>
      <td>${c.beds || '-'}</td>
      <td>${c.baths || '-'}</td>
      <td>${c.sqft || '-'}</td>
      <td>${c.soldDate || '-'}</td>
    </tr>`;
  });
  el.innerHTML = `
    <div class="panel">
      <div class="panel-body">
        <table class="table">
          <thead>
            <tr>
              <th>Address</th>
              <th>Distance</th>
              <th>Price</th>
              <th>Beds</th>
              <th>Baths</th>
              <th>Sq Ft</th>
              <th>Sold Date</th>
            </tr>
          </thead>
          <tbody>${rows}</tbody>
        </table>
      </div>
    </div>
  `;
}

function renderRentalComps(p) {
  const el = document.getElementById('rental-comps-body');
  const comps = (p.rentcast_rent && p.rentcast_rent.rentalComps) || [];
  if (!comps.length) {
    el.innerHTML = '<p class="help-block">No rental comparables available from RentCast.</p>';
    return;
  }
  let rows = '';
  comps.forEach(c => {
    rows += `<tr>
      <td>${c.address || ''}</td>
      <td>${c.distance ? c.distance.toFixed(2) + ' mi' : '-'}</td>
      <td>${formatCurrency(c.rent)}</td>
      <td>${c.beds || '-'}</td>
      <td>${c.baths || '-'}</td>
      <td>${c.sqft || '-'}</td>
      <td>${c.listedDate || '-'}</td>
    </tr>`;
  });
  el.innerHTML = `
    <div class="panel">
      <div class="panel-body">
        <table class="table">
          <thead>
            <tr>
              <th>Address</th>
              <th>Distance</th>
              <th>Rent</th>
              <th>Beds</th>
              <th>Baths</th>
              <th>Sq Ft</th>
              <th>Listed Date</th>
            </tr>
          </thead>
          <tbody>${rows}</tbody>
        </table>
      </div>
    </div>
  `;
}

function renderRecords(p) {
  const el = document.getElementById('records-body');
  const rc = p.rentcast_property || {};
  const lines = [
    `<strong>Public Records</strong>`,
    `Last Sale Price: ${formatCurrency(rc.lastSalePrice)}`,
    `Last Sale Date: ${rc.lastSaleDate || '-'}`,
    `Assessed Value: ${formatCurrency(rc.assessedValue)}`,
    `Assessed Year: ${rc.assessedYear || '-'}`,
    '',
    `<strong>Listing Info</strong>`,
    `MLS Status: ${rc.mlsStatus || '-'}`,
    `MLS ID: ${rc.mlsId || '-'}`,
    `Listing Price: ${formatCurrency(rc.listPrice)}`,
    `Days on Market: ${rc.daysOnMarket || '-'}`
  ];
  el.innerHTML = `
    <div class="panel">
      <div class="panel-body">
        <p>${lines.join('<br/>')}</p>
      </div>
    </div>
  `;
}

function renderOwner(p) {
  const el = document.getElementById('owner-body');
  const rc = p.rentcast_property || {};
  const lines = [
    `<strong>Owner Info (from RentCast where available)</strong>`,
    `Owner Name: ${rc.ownerName || '-'}`,
    `Owner Type: ${rc.ownerType || '-'}`,
    `Owner Occupied: ${rc.ownerOccupied == null ? '-' : (rc.ownerOccupied ? 'Yes' : 'No')}`,
    `Mailing Address: ${rc.ownerMailingAddress || '-'}`
  ];
  el.innerHTML = `
    <div class="panel">
      <div class="panel-body">
        <p>${lines.join('<br/>')}</p>
      </div>
    </div>
  `;
}

function renderOfferCalculator(p) {
  const el = document.getElementById('offer-body');
  const arv = p.est_value || 0;
  const repairs = p.rehab_budget || 0;
  const html = `
    <div class="panel">
      <div class="panel-body">
        <div class="summary-item-block">
          <div class="summary-line-item">
            <label>After Repair Value (ARV)</label>
            <input type="number" id="offer-arv" value="${arv}" />
          </div>
          <div class="summary-line-item">
            <label>Estimated Repairs</label>
            <input type="number" id="offer-repairs" value="${repairs}" />
          </div>
          <div class="summary-line-item">
            <label>Max % of ARV (flip)</label>
            <input type="number" id="offer-max-percent" value="70" />
          </div>
          <div class="summary-line-item">
            <label>Max % of ARV (rental)</label>
            <input type="number" id="offer-max-percent-rental" value="80" />
          </div>
          <div class="summary-line-item-total">
            <div>Max Flip Offer: <span id="offer-max-flip"></span></div>
            <div>Max Rental Offer: <span id="offer-max-rental"></span></div>
          </div>
        </div>
        <div class="margin-t-10">
          <button class="btn btn-primary" onclick="recalculateOffer()">Recalculate Offer</button>
        </div>
      </div>
    </div>
  `;
  el.innerHTML = html;
  recalculateOffer();
}

function recalculateOffer() {
  const arv = Number(document.getElementById('offer-arv').value) || 0;
  const repairs = Number(document.getElementById('offer-repairs').value) || 0;
  const maxPctFlip = Number(document.getElementById('offer-max-percent').value) || 0;
  const maxPctRental = Number(document.getElementById('offer-max-percent-rental').value) || 0;

  const maxFlip = (arv * (maxPctFlip / 100)) - repairs;
  const maxRental = (arv * (maxPctRental / 100)) - repairs;

  document.getElementById('offer-max-flip').textContent = formatCurrency(maxFlip);
  document.getElementById('offer-max-rental').textContent = formatCurrency(maxRental);
}

function renderReports(p) {
  const el = document.getElementById('reports-body');
  el.innerHTML = `
    <div class="panel">
      <div class="panel-body">
        <p class="help-block">
          Reports &amp; sharing will generate a printable summary of all sections for this property.
          For now, you can print this page directly from your browser.
        </p>
      </div>
    </div>
  `;
}

function renderCompare(p) {
  const el = document.getElementById('compare-body');
  const list = loadLocalProperties();
  let rows = '';
  list.forEach(prop => {
    rows += `<tr>
      <td><a href="property.html?id=${encodeURIComponent(prop.id)}">${prop.formatted_address || prop.address_line1 || ''}</a></td>
      <td>${formatCurrency(prop.est_value)}</td>
      <td>${formatCurrency(prop.est_rent)}</td>
      <td>${prop.beds || '-'}</td>
      <td>${prop.baths || '-'}</td>
      <td>${prop.sqft || '-'}</td>
      <td>${formatPercent(prop.cap_rate || 0)}</td>
    </tr>`;
  });
  el.innerHTML = `
    <div class="panel">
      <div class="panel-body">
        <table class="table">
          <thead>
            <tr>
              <th>Property</th>
              <th>Est. Value</th>
              <th>Est. Rent</th>
              <th>Beds</th>
              <th>Baths</th>
              <th>Sq Ft</th>
              <th>Cap Rate</th>
            </tr>
          </thead>
          <tbody>${rows}</tbody>
        </table>
      </div>
    </div>
  `;
}

function renderLenders(p) {
  const el = document.getElementById('lenders-body');
  el.innerHTML = `
    <div class="panel">
      <div class="panel-body">
        <p class="help-block">
          In the future, this section will show local lenders based on the property's location.
        </p>
      </div>
    </div>
  `;
}

function renderTrack(p) {
  const el = document.getElementById('track-body');
  el.innerHTML = `
    <div class="panel">
      <div class="panel-body">
        <p class="help-block">
          Tracking after purchase will be powered by RentCast's rental and value tracking.
          For now, MLS9 stores your last analysis and projections for this property.
        </p>
      </div>
    </div>
  `;
}

// Navigation actions

function switchSection(evt) {
  evt.preventDefault();
  const section = evt.currentTarget.getAttribute('data-section');
  if (!section) return;
  document.querySelectorAll('.content-section').forEach(s => s.classList.remove('active'));
  const active = document.getElementById('section-' + section);
  if (active) active.classList.add('active');
}

function goBackToList() {
  window.location.href = 'index.html';
}

function deleteProperty() {
  if (!currentProperty) return;
  if (!confirm('Delete this property from MLS9?')) return;
  const params = new URLSearchParams();
  params.append('action', 'delete');
  params.append('id', currentProperty.id);
  fetch('api/property.php', {
    method: 'POST',
    body: params
  })
    .then(r => r.json())
    .then(data => {
      if (data.success) {
        // Also remove from localStorage
        const list = loadLocalProperties().filter(p => p.id !== currentProperty.id);
        window.localStorage.setItem(MLS9_STORAGE_KEY, JSON.stringify(list));
        goBackToList();
      } else if (data.error) {
        alert('Error: ' + data.error);
      }
    })
    .catch(err => {
      console.error(err);
      alert('There was an error deleting this property.');
    });
}
