<?php
// photos.php
session_start();
require_once __DIR__ . '/config.php'; // must define $pdo (PDO) and $API_KEYS

// ---- Helpers ----
function getPropertyById(PDO $pdo, int $id): ?array {
    $stmt = $pdo->prepare("SELECT * FROM properties WHERE id = :id LIMIT 1");
    $stmt->execute([':id' => $id]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    return $row ?: null;
}

function getPhotosByPropertyId(PDO $pdo, int $id): array {
    $stmt = $pdo->prepare("
        SELECT id, property_id, url, caption, is_primary, source, created_at
        FROM property_photos
        WHERE property_id = :id
        ORDER BY is_primary DESC, created_at DESC, id DESC
    ");
    $stmt->execute([':id' => $id]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
}

function addPhoto(PDO $pdo, int $propertyId, string $url, string $caption = ''): int {
    $stmt = $pdo->prepare("
        INSERT INTO property_photos (property_id, url, caption, is_primary, source, created_at)
        VALUES (:property_id, :url, :caption, 0, 'manual', NOW())
    ");
    $stmt->execute([
        ':property_id' => $propertyId,
        ':url'         => $url,
        ':caption'     => $caption
    ]);
    return (int)$pdo->lastInsertId();
}

function setPrimaryPhoto(PDO $pdo, int $propertyId, int $photoId): void {
    $pdo->beginTransaction();
    $stmt = $pdo->prepare("UPDATE property_photos SET is_primary = 0 WHERE property_id = :pid");
    $stmt->execute([':pid' => $propertyId]);

    $stmt = $pdo->prepare("
        UPDATE property_photos
        SET is_primary = 1
        WHERE id = :id AND property_id = :pid
    ");
    $stmt->execute([':id' => $photoId, ':pid' => $propertyId]);
    $pdo->commit();
}

function deletePhoto(PDO $pdo, int $propertyId, int $photoId): ?string {
    // Get file URL first so we can delete it from disk
    $stmt = $pdo->prepare("
        SELECT url FROM property_photos
        WHERE id = :id AND property_id = :pid
        LIMIT 1
    ");
    $stmt->execute([':id' => $photoId, ':pid' => $propertyId]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$row) {
        return null;
    }

    $url = $row['url'];

    $stmt = $pdo->prepare("
        DELETE FROM property_photos
        WHERE id = :id AND property_id = :pid
        LIMIT 1
    ");
    $stmt->execute([':id' => $photoId, ':pid' => $propertyId]);

    return $url;
}

// ---- Get property_id ----
$propertyId = isset($_GET['property_id']) ? (int)$_GET['property_id'] : 0;
if ($propertyId <= 0) {
    http_response_code(400);
    echo "Missing or invalid property_id.";
    exit;
}

$property = getPropertyById($pdo, $propertyId);
if (!$property) {
    http_response_code(404);
    echo "Property not found.";
    exit;
}

// ---- Handle POST actions ----
$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'upload_photo' && isset($_FILES['photo'])) {
        $file = $_FILES['photo'];

        if ($file['error'] !== UPLOAD_ERR_OK) {
            $error = 'File upload failed.';
        } else {
            $allowedMime = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
            $finfo = finfo_open(FILEINFO_MIME_TYPE);
            $mime  = finfo_file($finfo, $file['tmp_name']);
            finfo_close($finfo);

            if (!in_array($mime, $allowedMime, true)) {
                $error = 'Only JPG, PNG, GIF, or WEBP files are allowed.';
            } else {
                $uploadBaseDir = __DIR__ . '/uploads/properties/' . $propertyId . '/';
                if (!is_dir($uploadBaseDir)) {
                    mkdir($uploadBaseDir, 0775, true);
                }

                $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
                $safeName = 'photo_' . time() . '_' . bin2hex(random_bytes(4)) . '.' . $ext;

                $destPath = $uploadBaseDir . $safeName;
                if (move_uploaded_file($file['tmp_name'], $destPath)) {
                    // Build public URL (adjust /uploads/ path if your document root differs)
                    $publicUrl = '/uploads/properties/' . $propertyId . '/' . $safeName;
                    $caption   = trim($_POST['caption'] ?? '');
                    addPhoto($pdo, $propertyId, $publicUrl, $caption);
                    $success = 'Photo uploaded successfully.';
                } else {
                    $error = 'Failed to move uploaded file.';
                }
            }
        }
    } elseif ($action === 'set_primary' && isset($_POST['photo_id'])) {
        $photoId = (int)$_POST['photo_id'];
        setPrimaryPhoto($pdo, $propertyId, $photoId);
        $success = 'Primary photo updated.';
    } elseif ($action === 'delete_photo' && isset($_POST['photo_id'])) {
        $photoId = (int)$_POST['photo_id'];
        $url = deletePhoto($pdo, $propertyId, $photoId);
        if ($url) {
            // try to delete file from disk
            $filePath = __DIR__ . $url;
            if (is_file($filePath)) {
                @unlink($filePath);
            }
            $success = 'Photo deleted.';
        } else {
            $error = 'Photo not found or already deleted.';
        }
    }
}

// Reload photos after any changes
$photos = getPhotosByPropertyId($pdo, $propertyId);
$primaryPhoto = $photos[0] ?? null;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Photos – <?= htmlspecialchars($property['full_address'] ?? 'Property #' . $propertyId) ?></title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <!-- You can replace this with your global CSS file -->
    <style>
        * { box-sizing: border-box; }
        body {
            margin: 0;
            font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
            background: #f3f4f6;
            color: #111827;
        }
        header {
            background: #111827;
            color: #f9fafb;
            padding: 14px 24px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }
        header h1 {
            margin: 0;
            font-size: 1.1rem;
            font-weight: 600;
        }
        header .breadcrumbs {
            font-size: .85rem;
            color: #9ca3af;
        }
        .layout {
            display: flex;
            min-height: calc(100vh - 56px);
        }
        nav.sidebar {
            width: 230px;
            background: #111827;
            color: #9ca3af;
            padding: 16px 12px;
        }
        nav.sidebar h2 {
            font-size: .75rem;
            text-transform: uppercase;
            letter-spacing: .08em;
            margin: 0 0 8px;
            color: #6b7280;
        }
        nav.sidebar a {
            display: block;
            padding: 8px 10px;
            border-radius: 8px;
            text-decoration: none;
            color: inherit;
            font-size: .9rem;
        }
        nav.sidebar a.active {
            background: #1f2937;
            color: #e5e7eb;
            font-weight: 500;
        }
        nav.sidebar a:hover {
            background: #1f2937;
            color: #e5e7eb;
        }
        .content {
            flex: 1;
            padding: 20px 24px 32px;
            overflow-x: hidden;
        }
        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            gap: 16px;
            margin-bottom: 16px;
        }
        .page-header-title {
            font-size: 1.3rem;
            font-weight: 600;
            color: #111827;
        }
        .page-header-subtitle {
            font-size: .9rem;
            color: #6b7280;
        }
        .pill {
            display: inline-flex;
            align-items: center;
            padding: 4px 10px;
            border-radius: 999px;
            font-size: .75rem;
            background: #e5e7eb;
            color: #374151;
        }
        .pill span.dot {
            width: 6px;
            height: 6px;
            border-radius: 999px;
            background: #10b981;
            margin-right: 6px;
        }
        .grid {
            display: grid;
            grid-template-columns: minmax(0, 2fr) minmax(0, 1fr);
            gap: 20px;
        }
        .card {
            background: #ffffff;
            border-radius: 14px;
            padding: 14px 16px 16px;
            box-shadow: 0 10px 25px rgba(15,23,42,0.06);
            border: 1px solid #e5e7eb;
        }
        .card-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 10px;
        }
        .card-header h3 {
            margin: 0;
            font-size: 0.95rem;
            font-weight: 600;
            color: #111827;
        }
        .card-header span {
            font-size: .8rem;
            color: #6b7280;
        }
        .primary-photo-container {
            border-radius: 12px;
            overflow: hidden;
            border: 1px solid #e5e7eb;
            background: #111827;
            position: relative;
            min-height: 260px;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .primary-photo-container img {
            width: 100%;
            height: 100%;
            object-fit: contain;
            background: #000;
        }
        .primary-photo-empty {
            color: #6b7280;
            font-size: .9rem;
        }
        .primary-photo-caption {
            padding-top: 8px;
            font-size: .85rem;
            color: #4b5563;
        }
        .badge-primary {
            position: absolute;
            left: 10px;
            top: 10px;
            padding: 4px 8px;
            font-size: .75rem;
            border-radius: 999px;
            background: rgba(16,185,129,0.9);
            color: #ecfdf5;
        }
        .thumbnail-strip {
            margin-top: 12px;
            display: flex;
            gap: 10px;
            overflow-x: auto;
            padding-bottom: 4px;
        }
        .thumb-card {
            min-width: 96px;
            max-width: 110px;
            border-radius: 10px;
            border: 1px solid #e5e7eb;
            background: #f9fafb;
            overflow: hidden;
            display: flex;
            flex-direction: column;
            font-size: .75rem;
        }
        .thumb-card img {
            width: 100%;
            height: 80px;
            object-fit: cover;
        }
        .thumb-body {
            padding: 6px 7px 7px;
        }
        .thumb-caption {
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            margin-bottom: 6px;
            color: #4b5563;
        }
        .thumb-actions {
            display: flex;
            gap: 4px;
        }
        .btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            padding: 6px 10px;
            border-radius: 999px;
            border: 1px solid transparent;
            font-size: .8rem;
            cursor: pointer;
            background: #111827;
            color: #f9fafb;
        }
        .btn-sm {
            padding: 4px 7px;
            font-size: .7rem;
        }
        .btn-outline {
            background: #ffffff;
            color: #111827;
            border-color: #d1d5db;
        }
        .btn-danger {
            background: #b91c1c;
            border-color: #b91c1c;
            color: #fef2f2;
        }
        .btn:disabled {
            opacity: 0.6;
            cursor: default;
        }
        form.inline {
            display: inline;
            margin: 0;
        }
        .upload-form {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }
        .field-group {
            display: flex;
            flex-direction: column;
            gap: 4px;
        }
        label {
            font-size: .8rem;
            color: #4b5563;
        }
        input[type="file"], input[type="text"] {
            font-size: .8rem;
            padding: 7px 8px;
            border-radius: 8px;
            border: 1px solid #d1d5db;
            background: #f9fafb;
        }
        .helper {
            font-size: .75rem;
            color: #9ca3af;
        }
        .alert {
            border-radius: 999px;
            padding: 6px 10px;
            font-size: .75rem;
            margin-bottom: 12px;
        }
        .alert-error {
            background: #fef2f2;
            color: #991b1b;
            border: 1px solid #fecaca;
        }
        .alert-success {
            background: #ecfdf5;
            color: #166534;
            border: 1px solid #bbf7d0;
        }
        @media (max-width: 960px) {
            .layout {
                flex-direction: column;
            }
            nav.sidebar {
                width: 100%;
                display: flex;
                flex-wrap: wrap;
                gap: 4px;
            }
            nav.sidebar a {
                display: inline-flex;
            }
            .grid {
                grid-template-columns: minmax(0, 1fr);
            }
        }
    </style>
</head>
<body>
<header>
    <div>
        <div class="breadcrumbs">MLS9 &nbsp;•&nbsp; Photos</div>
        <h1>
            Photos – <?= htmlspecialchars($property['full_address'] ?? 'Property #' . $propertyId) ?>
        </h1>
    </div>
    <div class="pill">
        <span class="dot"></span>
        Property ID #<?= (int)$propertyId ?>
    </div>
</header>

<div class="layout">

    <!-- Sidebar navigation (adjust links to match your file names) -->
    <nav class="sidebar">
        <h2>Property</h2>
        <a href="property.php?property_id=<?= (int)$propertyId ?>">Overview</a>
        <a href="purchase.html?property_id=<?= (int)$propertyId ?>">Purchase</a>
        <a href="photos.php?property_id=<?= (int)$propertyId ?>" class="active">Photos</a>
        <a href="map.php?property_id=<?= (int)$propertyId ?>">Map</a>
        <a href="analysis.html?property_id=<?= (int)$propertyId ?>">Analysis</a>
        <a href="projections.html?property_id=<?= (int)$propertyId ?>">Projections</a>
        <a href="sales-comps.html?property_id=<?= (int)$propertyId ?>">Sales Comps</a>
        <a href="rental-comps.html?property_id=<?= (int)$propertyId ?>">Rental Comps</a>
        <a href="records.html?property_id=<?= (int)$propertyId ?>">Records</a>
        <a href="owner-lookup.html?property_id=<?= (int)$propertyId ?>">Owner Lookup</a>
        <a href="offer-calculator.html?property_id=<?= (int)$propertyId ?>">Offer Calculator</a>
        <a href="reports.html?property_id=<?= (int)$propertyId ?>">Reports</a>
        <a href="compare.html?property_id=<?= (int)$propertyId ?>">Compare</a>
        <a href="lenders.html?property_id=<?= (int)$propertyId ?>">Lenders</a>
        <a href="track.html?property_id=<?= (int)$propertyId ?>">Track</a>
    </nav>

    <!-- Main content -->
    <main class="content">
        <div class="page-header">
            <div>
                <div class="page-header-title">Property Photos</div>
                <div class="page-header-subtitle">
                    Manage MLS, API, and manual images for this property. Set a primary cover photo and upload your own.
                </div>
            </div>
        </div>

        <?php if ($error): ?>
            <div class="alert alert-error"><?= htmlspecialchars($error) ?></div>
        <?php endif; ?>
        <?php if ($success): ?>
            <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
        <?php endif; ?>

        <div class="grid">
            <!-- Left: Primary photo + thumbnails -->
            <section class="card">
                <div class="card-header">
                    <h3>Photo Gallery</h3>
                    <span><?= count($photos) ?> photo<?= count($photos) === 1 ? '' : 's' ?></span>
                </div>

                <div class="primary-photo-container">
                    <?php if ($primaryPhoto): ?>
                        <span class="badge-primary">Primary</span>
                        <img src="<?= htmlspecialchars($primaryPhoto['url']) ?>"
                             alt="Primary photo for property">
                    <?php else: ?>
                        <div class="primary-photo-empty">
                            No photos yet. Upload a photo or sync from your MLS/API integration.
                        </div>
                    <?php endif; ?>
                </div>

                <?php if ($primaryPhoto && !empty($primaryPhoto['caption'])): ?>
                    <div class="primary-photo-caption">
                        <?= htmlspecialchars($primaryPhoto['caption']) ?>
                    </div>
                <?php endif; ?>

                <div class="thumbnail-strip">
                    <?php foreach ($photos as $photo): ?>
                        <div class="thumb-card">
                            <img src="<?= htmlspecialchars($photo['url']) ?>"
                                 alt="Photo #<?= (int)$photo['id'] ?>">
                            <div class="thumb-body">
                                <?php if (!empty($photo['caption'])): ?>
                                    <div class="thumb-caption" title="<?= htmlspecialchars($photo['caption']) ?>">
                                        <?= htmlspecialchars($photo['caption']) ?>
                                    </div>
                                <?php else: ?>
                                    <div class="thumb-caption" title="No caption">No caption</div>
                                <?php endif; ?>
                                <div class="thumb-actions">
                                    <?php if ((int)$photo['is_primary'] === 0): ?>
                                        <form method="post" class="inline">
                                            <input type="hidden" name="action" value="set_primary">
                                            <input type="hidden" name="photo_id"
                                                   value="<?= (int)$photo['id'] ?>">
                                            <button class="btn btn-sm btn-outline" type="submit">
                                                Set primary
                                            </button>
                                        </form>
                                    <?php else: ?>
                                        <button class="btn btn-sm btn-outline" type="button" disabled>
                                            Primary
                                        </button>
                                    <?php endif; ?>
                                    <form method="post" class="inline"
                                          onsubmit="return confirm('Delete this photo permanently?');">
                                        <input type="hidden" name="action" value="delete_photo">
                                        <input type="hidden" name="photo_id"
                                               value="<?= (int)$photo['id'] ?>">
                                        <button class="btn btn-sm btn-danger" type="submit">
                                            Delete
                                        </button>
                                    </form>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </section>

            <!-- Right: Upload form -->
            <section class="card">
                <div class="card-header">
                    <h3>Upload Photo</h3>
                    <span>JPG, PNG, GIF, WEBP</span>
                </div>

                <form method="post" enctype="multipart/form-data" class="upload-form">
                    <input type="hidden" name="action" value="upload_photo">
                    <div class="field-group">
                        <label for="photo">Select image</label>
                        <input type="file" name="photo" id="photo" required>
                        <div class="helper">
                            Recommended: 1600×900 or larger. Max ~10 MB (server limit).
                        </div>
                    </div>
                    <div class="field-group">
                        <label for="caption">Caption (optional)</label>
                        <input type="text" name="caption" id="caption"
                               placeholder="Front elevation, kitchen, living room, etc.">
                    </div>
                    <div>
                        <button class="btn" type="submit">Upload Photo</button>
                    </div>
                </form>

                <hr style="margin: 16px -16px 12px; border: none; border-top: 1px solid #e5e7eb;">

                <div style="font-size: .8rem; color: #6b7280;">
                    <strong>Coming soon:</strong> automatic photo import from MLS, Zillow-style
                    sources, and other APIs. All imported photos will appear here and can be
                    managed just like manual uploads.
                </div>
            </section>
        </div>
    </main>
</div>
</body>
</html>
