<?php
// includes/purchase.php

// ---- Bootstrapping & DB connection ----------------------------------------

$configPath = __DIR__ . '/../api/config.php';
if (file_exists($configPath)) {
    require_once $configPath;
}

// If config.php didn't already create $pdo, try to build it from constants.
$pdo = $pdo ?? null;

if (!$pdo && defined('DB_HOST') && defined('DB_NAME') && defined('DB_USER') && defined('DB_PASS')) {
    try {
        $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4';
        $pdo = new PDO($dsn, DB_USER, DB_PASS, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
    } catch (Exception $e) {
        $pdo = null;
    }
}

// ---- Default state ---------------------------------------------------------

$errors   = [];
$messages = [];

$property = [
    'id'               => null,
    'address'          => '',
    'city'             => '',
    'state'            => '',
    'zip'              => '',
    'county'           => '',
    'purchase_price'   => '',
    'arv'              => '',
    'rehab_budget'     => '',
    'closing_costs'    => '',
    'holding_months'   => '',
    'lead_source'      => '',
    'acquisition_type' => '',
    'seller_name'      => '',
    'seller_phone'     => '',
    'seller_email'     => '',
    'notes'            => '',
];

// ---- Handle POST: create / update property --------------------------------

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $property['id']               = isset($_POST['id']) ? trim($_POST['id']) : null;
    $property['address']          = trim($_POST['address'] ?? '');
    $property['city']             = trim($_POST['city'] ?? '');
    $property['state']            = trim($_POST['state'] ?? '');
    $property['zip']              = trim($_POST['zip'] ?? '');
    $property['county']           = trim($_POST['county'] ?? '');
    $property['purchase_price']   = trim($_POST['purchase_price'] ?? '');
    $property['arv']              = trim($_POST['arv'] ?? '');
    $property['rehab_budget']     = trim($_POST['rehab_budget'] ?? '');
    $property['closing_costs']    = trim($_POST['closing_costs'] ?? '');
    $property['holding_months']   = trim($_POST['holding_months'] ?? '');
    $property['lead_source']      = trim($_POST['lead_source'] ?? '');
    $property['acquisition_type'] = trim($_POST['acquisition_type'] ?? '');
    $property['seller_name']      = trim($_POST['seller_name'] ?? '');
    $property['seller_phone']     = trim($_POST['seller_phone'] ?? '');
    $property['seller_email']     = trim($_POST['seller_email'] ?? '');
    $property['notes']            = trim($_POST['notes'] ?? '');

    // Basic validation
    if ($property['address'] === '') {
        $errors[] = 'Address is required.';
    }

    if (!$pdo) {
        $errors[] = 'Database connection is not available.';
    }

    if (empty($errors) && $pdo) {
        try {
            if ($property['id']) {
                // UPDATE existing property
                $sql = "
                    UPDATE properties
                    SET
                        address          = :address,
                        city             = :city,
                        state            = :state,
                        zip              = :zip,
                        county           = :county,
                        purchase_price   = :purchase_price,
                        arv              = :arv,
                        rehab_budget     = :rehab_budget,
                        closing_costs    = :closing_costs,
                        holding_months   = :holding_months,
                        lead_source      = :lead_source,
                        acquisition_type = :acquisition_type,
                        seller_name      = :seller_name,
                        seller_phone     = :seller_phone,
                        seller_email     = :seller_email,
                        notes            = :notes,
                        updated_at       = NOW()
                    WHERE id = :id
                    LIMIT 1
                ";
            } else {
                // INSERT new property
                $sql = "
                    INSERT INTO properties (
                        address,
                        city,
                        state,
                        zip,
                        county,
                        purchase_price,
                        arv,
                        rehab_budget,
                        closing_costs,
                        holding_months,
                        lead_source,
                        acquisition_type,
                        seller_name,
                        seller_phone,
                        seller_email,
                        notes,
                        created_at,
                        updated_at
                    ) VALUES (
                        :address,
                        :city,
                        :state,
                        :zip,
                        :county,
                        :purchase_price,
                        :arv,
                        :rehab_budget,
                        :closing_costs,
                        :holding_months,
                        :lead_source,
                        :acquisition_type,
                        :seller_name,
                        :seller_phone,
                        :seller_email,
                        :notes,
                        NOW(),
                        NOW()
                    )
                ";
            }

            $stmt = $pdo->prepare($sql);

            if ($property['id']) {
                $stmt->bindValue(':id', $property['id'], PDO::PARAM_INT);
            }

            $stmt->bindValue(':address',          $property['address']);
            $stmt->bindValue(':city',             $property['city']);
            $stmt->bindValue(':state',            $property['state']);
            $stmt->bindValue(':zip',              $property['zip']);
            $stmt->bindValue(':county',           $property['county']);
            $stmt->bindValue(':purchase_price',   $property['purchase_price'] !== '' ? $property['purchase_price'] : null);
            $stmt->bindValue(':arv',              $property['arv'] !== '' ? $property['arv'] : null);
            $stmt->bindValue(':rehab_budget',     $property['rehab_budget'] !== '' ? $property['rehab_budget'] : null);
            $stmt->bindValue(':closing_costs',    $property['closing_costs'] !== '' ? $property['closing_costs'] : null);
            $stmt->bindValue(':holding_months',   $property['holding_months'] !== '' ? $property['holding_months'] : null);
            $stmt->bindValue(':lead_source',      $property['lead_source']);
            $stmt->bindValue(':acquisition_type', $property['acquisition_type']);
            $stmt->bindValue(':seller_name',      $property['seller_name']);
            $stmt->bindValue(':seller_phone',     $property['seller_phone']);
            $stmt->bindValue(':seller_email',     $property['seller_email']);
            $stmt->bindValue(':notes',            $property['notes']);

            $stmt->execute();

            if (!$property['id']) {
                $property['id'] = (int)$pdo->lastInsertId();
            }

            // After saving, go directly to analysis (root-level HTML that includes its own PHP)
            header('Location: ../analysis.html?id=' . urlencode($property['id']));
            exit;

        } catch (Exception $e) {
            $errors[] = 'Failed to save property. Please verify the database structure.';
        }
    }
}

// ---- Handle GET: load existing property or prefill address -----------------

if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $id      = isset($_GET['id']) ? trim($_GET['id']) : null;
    $address = isset($_GET['address']) ? trim($_GET['address']) : '';

    if ($id && $pdo) {
        try {
            $stmt = $pdo->prepare("
                SELECT
                    id,
                    address,
                    city,
                    state,
                    zip,
                    county,
                    purchase_price,
                    arv,
                    rehab_budget,
                    closing_costs,
                    holding_months,
                    lead_source,
                    acquisition_type,
                    seller_name,
                    seller_phone,
                    seller_email,
                    notes
                FROM properties
                WHERE id = :id
                LIMIT 1
            ");
            $stmt->execute([':id' => $id]);
            $row = $stmt->fetch();

            if ($row) {
                foreach ($property as $key => $val) {
                    if (array_key_exists($key, $row)) {
                        $property[$key] = $row[$key] ?? '';
                    }
                }
            }
        } catch (Exception $e) {
            // ignore; property fallback remains defaults
        }
    } elseif ($address !== '') {
        // Coming from index with an address only
        $property['address'] = $address;
        if (!empty($_GET['citystate'])) {
            $property['city'] = trim($_GET['citystate']);
        }
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>MLS9 Deal Analyzer | Purchase Details</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">

    <style>
        :root {
            --bg: #070b12;
            --bg-elevated: #0f172a;
            --bg-elevated-soft: #111827;
            --accent: #22c55e;
            --accent-soft: rgba(34,197,94,0.12);
            --accent-strong: rgba(34,197,94,0.24);
            --border-subtle: rgba(148,163,184,0.25);
            --text-main: #e5e7eb;
            --text-muted: #9ca3af;
            --danger: #f97373;
            --shadow-soft: 0 18px 45px rgba(15,23,42,0.75);
            --radius-lg: 18px;
            --radius-xl: 26px;
        }

        * {
            box-sizing: border-box;
        }

        body {
            margin: 0;
            font-family: system-ui, -apple-system, BlinkMacSystemFont, "SF Pro Text", "Segoe UI", sans-serif;
            background: radial-gradient(circle at top left, #1e293b 0, #020617 40%, #000 100%);
            color: var(--text-main);
            min-height: 100vh;
            display: flex;
        }

        a {
            color: inherit;
            text-decoration: none;
        }

        .layout {
            display: flex;
            width: 100%;
            min-height: 100vh;
        }

        .sidebar-shell {
            width: 260px;
            background: linear-gradient(180deg, #020617 0%, #020617 40%, #020617 100%);
            border-right: 1px solid rgba(15,23,42,0.9);
            box-shadow: 12px 0 40px rgba(0,0,0,0.6);
            position: relative;
            z-index: 10;
        }

        .main {
            flex: 1;
            display: flex;
            flex-direction: column;
            min-width: 0;
        }

        .topbar-shell {
            position: sticky;
            top: 0;
            z-index: 9;
            backdrop-filter: blur(16px);
            background: linear-gradient(90deg, rgba(15,23,42,0.9), rgba(15,23,42,0.75), rgba(15,23,42,0.9));
            border-bottom: 1px solid rgba(30,64,175,0.55);
        }

        .topbar-inner {
            max-width: 1240px;
            margin: 0 auto;
            padding: 14px 22px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            gap: 16px;
        }

        .brand-lockup {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .brand-mark {
            width: 34px;
            height: 34px;
            border-radius: 999px;
            border: 1px solid rgba(148,163,184,0.4);
            background: radial-gradient(circle at 20% 0, #22c55e 0, #0ea5e9 35%, #1d2435 100%);
            box-shadow: 0 0 0 1px rgba(15,23,42,0.9), 0 12px 30px rgba(34,197,94,0.35);
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 19px;
        }

        .brand-text {
            display: flex;
            flex-direction: column;
            gap: 2px;
        }

        .brand-text span:first-child {
            font-size: 14px;
            letter-spacing: 0.18em;
            text-transform: uppercase;
            color: #9ca3af;
        }

        .brand-text span:last-child {
            font-size: 18px;
            font-weight: 600;
        }

        .topbar-meta {
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 13px;
            color: var(--text-muted);
        }

        .pill {
            padding: 4px 10px;
            border-radius: 999px;
            border: 1px solid rgba(148,163,184,0.4);
            background: radial-gradient(circle at top left, rgba(34,197,94,0.2), transparent 60%);
            display: inline-flex;
            align-items: center;
            gap: 6px;
        }

        .pill-dot {
            width: 8px;
            height: 8px;
            border-radius: 999px;
            background: #22c55e;
            box-shadow: 0 0 0 4px rgba(34,197,94,0.35);
        }

        .content-wrap {
            max-width: 1240px;
            margin: 22px auto 40px;
            padding: 0 22px 22px;
            display: flex;
            flex-direction: column;
            gap: 18px;
        }

        .page-header {
            display: flex;
            align-items: flex-start;
            justify-content: space-between;
            gap: 14px;
        }

        .page-title {
            font-size: 22px;
            font-weight: 600;
        }

        .page-sub {
            font-size: 13px;
            color: var(--text-muted);
        }

        .badge {
            padding: 4px 9px;
            border-radius: 999px;
            background: rgba(15,23,42,0.8);
            border: 1px solid rgba(148,163,184,0.5);
            font-size: 11px;
            text-transform: uppercase;
            letter-spacing: 0.12em;
            color: var(--text-muted);
        }

        .card {
            border-radius: var(--radius-xl);
            background: radial-gradient(circle at top left, rgba(34,197,94,0.08), rgba(15,23,42,0.96));
            border: 1px solid var(--border-subtle);
            box-shadow: var(--shadow-soft);
            padding: 18px 18px 16px;
            position: relative;
            overflow: hidden;
        }

        .card::before {
            content: "";
            position: absolute;
            inset: -40%;
            pointer-events: none;
            background:
                radial-gradient(circle at 0 0, rgba(56,189,248,0.18), transparent 50%),
                radial-gradient(circle at 100% 0, rgba(34,197,94,0.16), transparent 50%);
            opacity: 0.6;
        }

        .card-inner {
            position: relative;
            z-index: 1;
        }

        .section-heading {
            font-size: 13px;
            text-transform: uppercase;
            letter-spacing: 0.14em;
            color: #9ca3af;
            margin-bottom: 6px;
        }

        .card-subtitle {
            font-size: 12px;
            color: var(--text-muted);
            margin-bottom: 12px;
        }

        .grid-main {
            display: grid;
            grid-template-columns: minmax(0, 1.4fr) minmax(0, 1fr);
            gap: 18px;
        }

        @media (max-width: 960px) {
            .sidebar-shell {
                display: none;
            }
            .grid-main {
                grid-template-columns: minmax(0, 1fr);
            }
        }

        .fields-grid {
            display: grid;
            grid-template-columns: repeat(2, minmax(0, 1fr));
            gap: 10px 12px;
        }

        @media (max-width: 720px) {
            .fields-grid {
                grid-template-columns: minmax(0, 1fr);
            }
        }

        .field {
            display: flex;
            flex-direction: column;
            gap: 3px;
        }

        .field-half-row {
            display: grid;
            grid-template-columns: repeat(2, minmax(0, 1fr));
            gap: 8px;
        }

        .label {
            font-size: 12px;
            color: var(--text-muted);
        }

        .label span.required {
            color: #f97373;
            margin-left: 2px;
        }

        .input,
        .select,
        .textarea {
            border-radius: 999px;
            padding: 8px 12px;
            border: 1px solid rgba(148,163,184,0.5);
            background: rgba(15,23,42,0.85);
            color: var(--text-main);
            font-size: 14px;
            outline: none;
            width: 100%;
        }

        .textarea {
            border-radius: 14px;
            min-height: 90px;
            resize: vertical;
            padding-top: 9px;
        }

        .input::placeholder,
        .textarea::placeholder {
            color: #6b7280;
        }

        .input:focus,
        .select:focus,
        .textarea:focus {
            border-color: #22c55e;
            box-shadow: 0 0 0 1px rgba(34,197,94,0.5);
        }

        .input[readonly] {
            opacity: 0.9;
        }

        .input-group-inline {
            display: flex;
            gap: 8px;
            align-items: center;
        }

        .prefix {
            font-size: 13px;
            color: #9ca3af;
            padding: 0 4px;
        }

        .btn-primary {
            border-radius: 999px;
            padding: 9px 18px;
            font-size: 14px;
            font-weight: 500;
            border: none;
            cursor: pointer;
            background: linear-gradient(135deg, #22c55e, #16a34a);
            color: #020617;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 7px;
            white-space: nowrap;
        }

        .btn-secondary {
            border-radius: 999px;
            padding: 8px 14px;
            font-size: 13px;
            border: 1px solid rgba(148,163,184,0.6);
            cursor: pointer;
            background: rgba(15,23,42,0.9);
            color: var(--text-main);
            display: inline-flex;
            align-items: center;
            gap: 6px;
        }

        .btn-tertiary {
            border-radius: 999px;
            padding: 6px 12px;
            font-size: 12px;
            border: 1px dashed rgba(148,163,184,0.6);
            cursor: pointer;
            background: rgba(15,23,42,0.7);
            color: var(--text-muted);
            display: inline-flex;
            align-items: center;
            gap: 6px;
        }

        .btn-primary span.icon,
        .btn-secondary span.icon {
            font-size: 15px;
        }

        .actions-row {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            justify-content: flex-end;
            margin-top: 16px;
        }

        .pill-small {
            border-radius: 999px;
            padding: 3px 8px;
            border: 1px solid rgba(148,163,184,0.6);
            font-size: 11px;
            color: var(--text-muted);
        }

        .summary-grid {
            display: grid;
            grid-template-columns: repeat(2, minmax(0, 1fr));
            gap: 10px;
            margin-top: 10px;
            font-size: 13px;
        }

        .summary-item {
            border-radius: 12px;
            padding: 8px 10px;
            border: 1px solid rgba(148,163,184,0.5);
            background: rgba(15,23,42,0.92);
        }

        .summary-label {
            font-size: 11px;
            color: var(--text-muted);
            text-transform: uppercase;
            letter-spacing: 0.12em;
            margin-bottom: 2px;
        }

        .summary-value {
            font-size: 15px;
            font-weight: 500;
        }

        .summary-note {
            font-size: 11px;
            color: var(--text-muted);
            margin-top: 2px;
        }

        .bad {
            color: #f97373;
        }

        .good {
            color: #4ade80;
        }

        .messages {
            margin-top: 4px;
        }

        .error-box,
        .info-box {
            border-radius: 12px;
            padding: 8px 11px;
            font-size: 12px;
            margin-bottom: 6px;
        }

        .error-box {
            border: 1px solid rgba(248,113,113,0.75);
            background: rgba(127,29,29,0.7);
        }

        .info-box {
            border: 1px solid rgba(56,189,248,0.75);
            background: rgba(15,23,42,0.85);
        }

        .error-box ul {
            margin: 0;
            padding-left: 18px;
        }

        .error-box li {
            margin-bottom: 3px;
        }

        @media (max-width: 540px) {
            .page-header {
                flex-direction: column;
                align-items: flex-start;
            }
            .actions-row {
                justify-content: stretch;
            }
            .btn-primary,
            .btn-secondary {
                flex: 1;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
<div class="layout">

    <!-- Sidebar -->
    <aside class="sidebar-shell">
        <?php
        $sidebarPath = __DIR__ . '/sidebar.php'; // your existing menu/side nav in includes/
        if (file_exists($sidebarPath)) {
            include $sidebarPath;
        }
        ?>
    </aside>

    <div class="main">
        <!-- Topbar -->
        <header class="topbar-shell">
            <div class="topbar-inner">
                <div class="brand-lockup">
                    <div class="brand-mark">9</div>
                    <div class="brand-text">
                        <span>MLS9 | REI9</span>
                        <span>Purchase details</span>
                    </div>
                </div>
                <div class="topbar-meta">
                    <div class="pill">
                        <span class="pill-dot"></span>
                        <span>Lock in the deal right from day one</span>
                    </div>
                </div>
            </div>
        </header>

        <!-- Content -->
        <main class="content-wrap">
            <div class="page-header">
                <div>
                    <div class="page-title">
                        Purchase details for
                        <?php if (!empty($property['address'])): ?>
                            “<?= htmlspecialchars($property['address']) ?>”
                        <?php else: ?>
                            a new property
                        <?php endif; ?>
                    </div>
                    <div class="page-sub">
                        Set up the entry side of the deal: purchase price, rehab, closing costs, and lead/source context.
                        Everything here flows into analysis, projections, and reports.
                    </div>
                </div>
                <div>
                    <span class="badge">
                        Step 1 of pipeline &mdash; Purchase
                    </span>
                </div>
            </div>

            <section class="card">
                <div class="card-inner">

                    <?php if (!empty($errors)): ?>
                        <div class="messages">
                            <div class="error-box">
                                <strong>Fix the following:</strong>
                                <ul>
                                    <?php foreach ($errors as $error): ?>
                                        <li><?= htmlspecialchars($error) ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        </div>
                    <?php endif; ?>

                    <?php if (!empty($messages)): ?>
                        <div class="messages">
                            <?php foreach ($messages as $msg): ?>
                                <div class="info-box"><?= htmlspecialchars($msg) ?></div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>

                    <form id="purchase-form" method="post" action="purchase.php">
                        <input type="hidden" name="id" id="prop-id" value="<?= htmlspecialchars($property['id'] ?? '') ?>">

                        <div class="grid-main">
                            <!-- LEFT: Property & Numbers -->
                            <div>
                                <div class="section-heading">Property &amp; purchase numbers</div>
                                <div class="card-subtitle">
                                    Start with the basics. You can refine ARV, rehab, and strategy later in the analysis tab.
                                </div>

                                <div class="fields-grid">
                                    <div class="field">
                                        <label class="label" for="address">
                                            Property address<span class="required">*</span>
                                        </label>
                                        <input
                                            class="input"
                                            id="address"
                                            name="address"
                                            type="text"
                                            required
                                            placeholder="Street address"
                                            value="<?= htmlspecialchars($property['address']) ?>"
                                        >
                                    </div>

                                    <div class="field">
                                        <label class="label" for="city">City</label>
                                        <input
                                            class="input"
                                            id="city"
                                            name="city"
                                            type="text"
                                            placeholder="City"
                                            value="<?= htmlspecialchars($property['city']) ?>"
                                        >
                                    </div>

                                    <div class="field field-half-row">
                                        <div>
                                            <label class="label" for="state">State</label>
                                            <input
                                                class="input"
                                                id="state"
                                                name="state"
                                                type="text"
                                                placeholder="ST"
                                                maxlength="2"
                                                value="<?= htmlspecialchars($property['state']) ?>"
                                            >
                                        </div>
                                        <div>
                                            <label class="label" for="zip">ZIP</label>
                                            <input
                                                class="input"
                                                id="zip"
                                                name="zip"
                                                type="text"
                                                placeholder="ZIP"
                                                value="<?= htmlspecialchars($property['zip']) ?>"
                                            >
                                        </div>
                                    </div>

                                    <div class="field">
                                        <label class="label" for="county">County (optional)</label>
                                        <input
                                            class="input"
                                            id="county"
                                            name="county"
                                            type="text"
                                            placeholder="County"
                                            value="<?= htmlspecialchars($property['county']) ?>"
                                        >
                                    </div>

                                    <div class="field">
                                        <label class="label" for="purchase_price">Purchase price</label>
                                        <div class="input-group-inline">
                                            <span class="prefix">$</span>
                                            <input
                                                class="input"
                                                id="purchase_price"
                                                name="purchase_price"
                                                type="number"
                                                step="0.01"
                                                min="0"
                                                placeholder="0"
                                                value="<?= htmlspecialchars($property['purchase_price']) ?>"
                                            >
                                        </div>
                                    </div>

                                    <div class="field">
                                        <label class="label" for="arv">ARV (After Repair Value)</label>
                                        <div class="input-group-inline">
                                            <span class="prefix">$</span>
                                            <input
                                                class="input"
                                                id="arv"
                                                name="arv"
                                                type="number"
                                                step="0.01"
                                                min="0"
                                                placeholder="0"
                                                value="<?= htmlspecialchars($property['arv']) ?>"
                                            >
                                        </div>
                                    </div>

                                    <div class="field">
                                        <label class="label" for="rehab_budget">Rehab / repairs budget</label>
                                        <div class="input-group-inline">
                                            <span class="prefix">$</span>
                                            <input
                                                class="input"
                                                id="rehab_budget"
                                                name="rehab_budget"
                                                type="number"
                                                step="0.01"
                                                min="0"
                                                placeholder="0"
                                                value="<?= htmlspecialchars($property['rehab_budget']) ?>"
                                            >
                                        </div>
                                    </div>

                                    <div class="field">
                                        <label class="label" for="closing_costs">Closing &amp; entry costs</label>
                                        <div class="input-group-inline">
                                            <span class="prefix">$</span>
                                            <input
                                                class="input"
                                                id="closing_costs"
                                                name="closing_costs"
                                                type="number"
                                                step="0.01"
                                                min="0"
                                                placeholder="0"
                                                value="<?= htmlspecialchars($property['closing_costs']) ?>"
                                            >
                                        </div>
                                    </div>

                                    <div class="field">
                                        <label class="label" for="holding_months">Estimated hold period (months)</label>
                                        <input
                                            class="input"
                                            id="holding_months"
                                            name="holding_months"
                                            type="number"
                                            step="1"
                                            min="0"
                                            placeholder="0"
                                            value="<?= htmlspecialchars($property['holding_months']) ?>"
                                        >
                                    </div>

                                    <div class="field">
                                        <label class="label" for="lead_source">Lead source</label>
                                        <input
                                            class="input"
                                            id="lead_source"
                                            name="lead_source"
                                            type="text"
                                            placeholder="e.g. PPC, direct mail, JV, MLS, cold call"
                                            value="<?= htmlspecialchars($property['lead_source']) ?>"
                                        >
                                    </div>

                                    <div class="field">
                                        <label class="label" for="acquisition_type">Planned acquisition type</label>
                                        <select class="select" id="acquisition_type" name="acquisition_type">
                                            <option value="">Select...</option>
                                            <?php
                                            $types = [
                                                'Cash Purchase',
                                                'Wholesale Assignment',
                                                'Subject To',
                                                'Seller Finance',
                                                'Novation Agreement',
                                                'Wrap Mortgage',
                                                'BRRRR',
                                                'Lease Option',
                                                'Other Creative',
                                            ];
                                            foreach ($types as $type):
                                                $selected = ($property['acquisition_type'] === $type) ? 'selected' : '';
                                                ?>
                                                <option value="<?= htmlspecialchars($type) ?>" <?= $selected ?>>
                                                    <?= htmlspecialchars($type) ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                </div>

                                <div style="margin-top: 10px; display:flex; gap:8px; flex-wrap:wrap;">
                                    <button type="button" class="btn-secondary" id="btn-api-fill">
                                        <span class="icon">⚡</span>
                                        Pull data from API
                                    </button>
                                    <span class="pill-small">
                                        Uses your property data APIs (RentCast / etc.) via <code>/api/property-autofill.php</code>.
                                    </span>
                                </div>
                            </div>

                            <!-- RIGHT: Seller & Summary -->
                            <div>
                                <div class="section-heading">Seller &amp; quick summary</div>
                                <div class="card-subtitle">
                                    Capture seller context and see a real-time snapshot of total cost, spread, and basic equity.
                                </div>

                                <div class="fields-grid">
                                    <div class="field">
                                        <label class="label" for="seller_name">Seller name</label>
                                        <input
                                            class="input"
                                            id="seller_name"
                                            name="seller_name"
                                            type="text"
                                            placeholder="Owner / decision maker"
                                            value="<?= htmlspecialchars($property['seller_name']) ?>"
                                        >
                                    </div>

                                    <div class="field">
                                        <label class="label" for="seller_phone">Seller phone</label>
                                        <input
                                            class="input"
                                            id="seller_phone"
                                            name="seller_phone"
                                            type="text"
                                            placeholder="(xxx) xxx-xxxx"
                                            value="<?= htmlspecialchars($property['seller_phone']) ?>"
                                        >
                                    </div>

                                    <div class="field">
                                        <label class="label" for="seller_email">Seller email</label>
                                        <input
                                            class="input"
                                            id="seller_email"
                                            name="seller_email"
                                            type="email"
                                            placeholder="name@example.com"
                                            value="<?= htmlspecialchars($property['seller_email']) ?>"
                                        >
                                    </div>

                                    <div class="field">
                                        <label class="label" for="notes">Notes / seller motivation</label>
                                        <textarea
                                            class="textarea"
                                            id="notes"
                                            name="notes"
                                            placeholder="Pain points, time frame, needs, objections..."
                                        ><?= htmlspecialchars($property['notes']) ?></textarea>
                                    </div>
                                </div>

                                <div class="summary-grid">
                                    <div class="summary-item">
                                        <div class="summary-label">Total project cost</div>
                                        <div class="summary-value" id="summary-total-cost">$0</div>
                                        <div class="summary-note">
                                            Purchase + rehab + closing.
                                        </div>
                                    </div>
                                    <div class="summary-item">
                                        <div class="summary-label">Equity at ARV</div>
                                        <div class="summary-value" id="summary-equity">$0</div>
                                        <div class="summary-note" id="summary-equity-note">
                                            Based on ARV minus total project cost.
                                        </div>
                                    </div>
                                    <div class="summary-item">
                                        <div class="summary-label">Cost / ARV %</div>
                                        <div class="summary-value" id="summary-cost-percent">0%</div>
                                        <div class="summary-note">
                                            Your total cost as a % of ARV.
                                        </div>
                                    </div>
                                    <div class="summary-item">
                                        <div class="summary-label">Quick signal</div>
                                        <div class="summary-value" id="summary-signal">–</div>
                                        <div class="summary-note">
                                            Colored signal as a first-pass gut check.
                                        </div>
                                    </div>
                                </div>

                                <div style="margin-top: 8px; font-size:11px; color:var(--text-muted);">
                                    Detailed loan terms, cash-on-cash, and exit strategy performance will be modeled
                                    on the <strong>Analysis</strong> and <strong>Projections</strong> pages.
                                </div>
                            </div>
                        </div>

                        <div class="actions-row">
                            <a href="../index.php" class="btn-tertiary">← Back to dashboard</a>
                            <button type="submit" class="btn-primary">
                                <span class="icon">💾</span>
                                Save &amp; continue to analysis
                            </button>
                        </div>
                    </form>
                </div>
            </section>
        </main>
    </div>
</div>

<script>
// ---- Helper: format currency & percent -------------------------------------
function parseNumber(value) {
    if (typeof value === 'number') return value;
    const v = String(value || '').replace(/[^0-9.-]/g, '');
    const n = parseFloat(v);
    return isNaN(n) ? 0 : n;
}

function formatCurrency(n) {
    return n.toLocaleString(undefined, { style: 'currency', currency: 'USD', maximumFractionDigits: 0 });
}

function formatPercent(n) {
    return n.toLocaleString(undefined, { maximumFractionDigits: 1 }) + '%';
}

// ---- Live summary calculation ---------------------------------------------
function recalcSummary() {
    const purchase = parseNumber(document.getElementById('purchase_price').value);
    const rehab    = parseNumber(document.getElementById('rehab_budget').value);
    const closing  = parseNumber(document.getElementById('closing_costs').value);
    const arv      = parseNumber(document.getElementById('arv').value);

    const totalCost = purchase + rehab + closing;
    const equity    = arv - totalCost;

    const totalCostEl   = document.getElementById('summary-total-cost');
    const equityEl      = document.getElementById('summary-equity');
    const equityNoteEl  = document.getElementById('summary-equity-note');
    const costPctEl     = document.getElementById('summary-cost-percent');
    const signalEl      = document.getElementById('summary-signal');

    totalCostEl.textContent = formatCurrency(totalCost);
    equityEl.textContent    = formatCurrency(equity);

    let costPct = 0;
    if (arv > 0) {
        costPct = (totalCost / arv) * 100;
    }
    costPctEl.textContent = formatPercent(costPct);

    let signal = 'Need data';
    let note   = 'Based on basic cost vs ARV.';

    signalEl.classList.remove('good', 'bad');

    if (arv <= 0 || totalCost <= 0) {
        signal = 'Enter ARV & costs';
        note   = 'We’ll show a quick signal once ARV and cost are set.';
    } else if (costPct <= 70) {
        signal = 'Strong spread';
        note   = 'Total cost is at or under the classic 70% of ARV rule of thumb.';
        signalEl.classList.add('good');
    } else if (costPct <= 80) {
        signal = 'Tight but workable';
        note   = 'Total cost is under ~80% of ARV. Long-term holds or creative finance may still work.';
    } else {
        signal = 'Very tight';
        note   = 'Total cost is above 80% of ARV. You may need a better price, a different structure, or a higher ARV.';
        signalEl.classList.add('bad');
    }

    signalEl.textContent = signal;
    equityNoteEl.textContent = note;
}

['purchase_price','rehab_budget','closing_costs','arv'].forEach(id => {
    const el = document.getElementById(id);
    if (el) {
        el.addEventListener('input', recalcSummary);
    }
});

// Initial calc on load
recalcSummary();

// ---- API Autofill stub -----------------------------------------------------
document.getElementById('btn-api-fill').addEventListener('click', function () {
    const address = document.getElementById('address').value.trim();
    const city    = document.getElementById('city').value.trim();
    const state   = document.getElementById('state').value.trim();
    const zip     = document.getElementById('zip').value.trim();

    if (!address) {
        alert('Enter an address first so we know what to look up.');
        return;
    }

    const params = new URLSearchParams({
        address: address,
        city: city,
        state: state,
        zip: zip
    });

    // TODO: Implement /api/property-autofill.php to call your external APIs
    fetch('../api/property-autofill.php?' + params.toString(), {
        method: 'GET',
        headers: { 'Accept': 'application/json' }
    })
        .then(res => {
            if (!res.ok) throw new Error('API error');
            return res.json();
        })
        .then(data => {
            if (data.address && !document.getElementById('address').value) {
                document.getElementById('address').value = data.address;
            }
            if (data.city)   document.getElementById('city').value   = data.city;
            if (data.state)  document.getElementById('state').value  = data.state;
            if (data.zip)    document.getElementById('zip').value    = data.zip;
            if (data.county) document.getElementById('county').value = data.county;

            if (typeof data.arv !== 'undefined' && !document.getElementById('arv').value) {
                document.getElementById('arv').value = data.arv;
            }
            if (typeof data.rehab_estimate !== 'undefined' && !document.getElementById('rehab_budget').value) {
                document.getElementById('rehab_budget').value = data.rehab_estimate;
            }
            if (typeof data.purchase_price_suggestion !== 'undefined' && !document.getElementById('purchase_price').value) {
                document.getElementById('purchase_price').value = data.purchase_price_suggestion;
            }

            recalcSummary();
        })
        .catch(() => {
            alert('Could not pull data from API. Double check that api/property-autofill.php is implemented.');
        });
});
</script>

</body>
</html>
