// js/property.js

// Load Google Charts
google.charts.load('current', { packages: ['corechart'] });

document.addEventListener('DOMContentLoaded', () => {
  const params = new URLSearchParams(window.location.search);
  const propertyId = params.get('id');

  if (!propertyId) {
    alert('No property id specified in ?id=');
    window.location.href = 'index.html';
    return;
  }

  // Fetch property data from backend
  fetch(`api/property.php?action=get&id=${encodeURIComponent(propertyId)}`)
    .then(res => {
      if (!res.ok) throw new Error('Failed to fetch property');
      return res.json();
    })
    .then(data => {
      if (!data || !data.property) {
        throw new Error('Invalid property response');
      }
      const p = data.property;
      renderProperty(p);
    })
    .catch(err => {
      console.error(err);
      alert('Unable to load property. Please try again.');
    });

  // Monthly/Yearly toggle
  document.getElementById('btn-monthly').addEventListener('click', () => {
    document.getElementById('btn-monthly').classList.add('blue');
    document.getElementById('btn-yearly').classList.remove('blue');
    setCashFlowMode('monthly');
  });

  document.getElementById('btn-yearly').addEventListener('click', () => {
    document.getElementById('btn-yearly').classList.add('blue');
    document.getElementById('btn-monthly').classList.remove('blue');
    setCashFlowMode('yearly');
  });

  // For now, expenses chart is always loaded; buttons are here for future toggles
  document.getElementById('btn-expenses-chart').addEventListener('click', () => {
    // no-op, default is expenses
  });
  document.getElementById('btn-income-chart').addEventListener('click', () => {
    // could switch to income chart later
  });

  // Delete button shell
  const deleteBtn = document.getElementById('delete-property-btn');
  deleteBtn.addEventListener('click', () => {
    if (!confirm('Delete this property?')) return;
    fetch(`api/property.php?action=delete&id=${encodeURIComponent(propertyId)}`, {
      method: 'POST'
    }).then(res => res.json())
      .then(resp => {
        if (resp && resp.success) {
          alert('Property deleted.');
          window.location.href = 'index.html';
        } else {
          alert('Error deleting property.');
        }
      }).catch(err => {
        console.error(err);
        alert('Error deleting property.');
      });
  });
});

let currentMode = 'monthly'; // or 'yearly'
let cachedMetrics = null;

function setCashFlowMode(mode) {
  if (!cachedMetrics) return;
  currentMode = mode;
  const m = cachedMetrics;

  const factor = (mode === 'yearly') ? 12 : 1;

  setText('gross-rent', formatCurrency(m.rent * factor));
  setText('vacancy', formatCurrency(m.vacancy * factor));
  setText('operating-income', formatCurrency(m.operatingIncome * factor));
  setText('operating-expenses', formatCurrency(m.operatingExpenses * factor));
  setText('noi', formatCurrency(m.noi * factor));
  setText('loan-payments', formatCurrency(m.loanPayment * factor));
  const cf = (m.noi - m.loanPayment) * factor;
  setText('cash-flow', formatCurrency(cf));

  // Color red if negative
  const row = document.getElementById('cash-flow-row');
  if (cf < 0) {
    row.classList.add('red');
  } else {
    row.classList.remove('red');
  }

  // Post-tax (simple example: no tax adjustment)
  setText('post-tax-cash-flow', formatCurrency(cf));
}

/**
 * p = property JSON from backend.
 * We expect (you can adjust on backend):
 * {
 *   id,
 *   address, city, state, zip,
 *   beds, baths, sqft,
 *   photo_url,
 *   purchase_price,
 *   arv,
 *   rent,
 *   vacancy_rate,        // 0.1 for 10%
 *   taxes_monthly,
 *   insurance_monthly,
 *   management_monthly,
 *   maintenance_monthly,
 *   capex_monthly,
 *   hoa_monthly,
 *   utilities_monthly,
 *   other_expenses_monthly,
 *   closing_costs,
 *   rehab_costs,
 *   loan_amount,
 *   interest_rate,       // e.g. 7 for 7%
 *   loan_term_years      // e.g. 30
 * }
 */
function renderProperty(p) {
  // Basic identity
  const fullAddressLine = p.address || '';
  const cityLine = [p.city, p.state, p.zip].filter(Boolean).join(', ');
  const bedsText = p.beds != null ? p.beds + ' BR' : '';
  const bathsText = p.baths != null ? p.baths + ' BA' : '';
  const sqftText = p.sqft ? formatNumber(p.sqft) + ' Sq.Ft.' : '';

  // Sidebar
  setText('sidebar-address-line', fullAddressLine);
  setHtml('sidebar-address-block',
    `${escapeHtml(fullAddressLine)}<br>${escapeHtml(cityLine)}`);
  setText('sidebar-property-meta', [bedsText, bathsText, sqftText].filter(Boolean).join(' · '));
  setText('sidebar-price', formatCurrency(p.purchase_price || 0));
  setText('breadcrumb-address', fullAddressLine);

  if (p.photo_url) {
    const img = document.getElementById('property-photo');
    img.src = p.photo_url;
  }

  // Financing basics
  const purchasePrice = toNumber(p.purchase_price);
  const loanAmount = toNumber(p.loan_amount);
  const closingCosts = toNumber(p.closing_costs);
  const rehabCosts = toNumber(p.rehab_costs);
  const rent = toNumber(p.rent);
  const vacancyRate = (p.vacancy_rate != null) ? Number(p.vacancy_rate) : 0.1;
  const taxes = toNumber(p.taxes_monthly);
  const insurance = toNumber(p.insurance_monthly);
  const management = toNumber(p.management_monthly);
  const maintenance = toNumber(p.maintenance_monthly);
  const capex = toNumber(p.capex_monthly);
  const hoa = toNumber(p.hoa_monthly);
  const utilities = toNumber(p.utilities_monthly);
  const otherExp = toNumber(p.other_expenses_monthly);
  const arv = toNumber(p.arv);

  const downPayment = Math.max(purchasePrice - loanAmount, 0);
  const totalCashNeeded = downPayment + closingCosts + rehabCosts;

  // Quick stats
  setText('stat-cash-needed', formatCurrency(totalCashNeeded));
  setText('purchase-price', formatCurrency(purchasePrice));
  setText('amount-financed', formatCurrency(loanAmount));
  setText('down-payment', formatCurrency(downPayment));
  setText('purchase-costs', formatCurrency(closingCosts));
  setText('rehab-costs', formatCurrency(rehabCosts));
  setText('total-cash-needed', formatCurrency(totalCashNeeded));

  setText('loan-amount', formatCurrency(loanAmount));

  const ltc = purchasePrice > 0 ? (loanAmount / (purchasePrice + rehabCosts)) * 100 : 0;
  const ltv = arv > 0 ? (loanAmount / arv) * 100 : 0;
  setText('ltc', formatPercent(ltc));
  setText('ltv', formatPercent(ltv));

  // Show "Price (80%)" style text
  setText('financing-of', `Price (${formatPercent(ltc)})`);

  // Loan terms
  const rate = Number(p.interest_rate) || 0;
  const termYears = Number(p.loan_term_years) || 30;
  setText('interest-rate', formatPercent(rate));
  setText('loan-type', `Amortizing, ${termYears} Year`);

  // ARV & per-square-foot
  setText('arv', formatCurrency(arv));
  const sqft = toNumber(p.sqft);
  const arvPsf = (sqft > 0 && arv > 0) ? arv / sqft : 0;
  const pricePsf = (sqft > 0 && purchasePrice > 0) ? purchasePrice / sqft : 0;
  const rehabPsf = (sqft > 0 && rehabCosts > 0) ? rehabCosts / sqft : 0;

  setText('arv-psf', formatCurrency(arvPsf));
  setText('price-psf', formatCurrency(pricePsf));
  setText('rehab-psf', formatCurrency(rehabPsf));

  // Cash flow / Operating
  const grossRent = rent;
  const vacancy = grossRent * vacancyRate;
  const operatingIncome = grossRent - vacancy;
  const operatingExpenses = taxes + insurance + management + maintenance + capex + hoa + utilities + otherExp;
  const noi = operatingIncome - operatingExpenses;

  setText('gross-rent', formatCurrency(grossRent));
  setText('vacancy', formatCurrency(vacancy));
  setText('vacancy-percent', `${formatPercent(vacancyRate * 100)} of Rent`);
  setText('operating-income', formatCurrency(operatingIncome));
  setText('operating-expenses', formatCurrency(operatingExpenses));

  const expenseRatio = operatingIncome > 0 ? (operatingExpenses / operatingIncome) * 100 : 0;
  setText('expense-ratio', `${formatPercent(expenseRatio)} of Income`);

  setText('noi', formatCurrency(noi));

  // Loan payment (simple amortization)
  const monthlyRate = rate > 0 ? (rate / 100) / 12 : 0;
  const nPayments = termYears * 12;
  let loanPayment = 0;
  if (monthlyRate > 0 && nPayments > 0 && loanAmount > 0) {
    loanPayment = loanAmount * (monthlyRate * Math.pow(1 + monthlyRate, nPayments)) /
      (Math.pow(1 + monthlyRate, nPayments) - 1);
  }

  setText('loan-payments', formatCurrency(loanPayment));

  const monthlyCashFlow = noi - loanPayment;
  setText('cash-flow', formatCurrency(monthlyCashFlow));
  const cashFlowRow = document.getElementById('cash-flow-row');
  if (monthlyCashFlow < 0) cashFlowRow.classList.add('red');

  setText('post-tax-cash-flow', formatCurrency(monthlyCashFlow)); // simple for now

  // Cache metrics for monthly/yearly toggle
  cachedMetrics = {
    rent: grossRent,
    vacancy,
    operatingIncome,
    operatingExpenses,
    noi,
    loanPayment
  };

  // Default display monthly
  setCashFlowMode('monthly');

  // Returns
  const capRatePurchase = purchasePrice > 0 ? (noi * 12 / purchasePrice) * 100 : 0;
  const capRateMV = arv > 0 ? (noi * 12 / arv) * 100 : 0;
  setText('cap-rate', formatPercent(capRatePurchase));
  setText('cap-rate-mv', formatPercent(capRateMV));

  const yearlyCF = monthlyCashFlow * 12;
  const coc = totalCashNeeded > 0 ? (yearlyCF / totalCashNeeded) * 100 : 0;
  setText('coc', formatPercent(coc));

  // ROI/ROE/IRR placeholders – if you compute on backend, just return them instead
  setText('roe', formatPercent(0)); // can compute from equity later
  setText('roi', formatPercent(0));
  setText('irr', formatPercent(0));

  const rtv = purchasePrice > 0 ? (grossRent / purchasePrice) * 100 : 0;
  setText('rtv', formatPercent(rtv));

  const grm = grossRent > 0 ? (purchasePrice / (grossRent * 12)) : 0;
  setText('grm', grm.toFixed(2));

  // Equity multiple placeholder
  setText('equity-multiple', '0.00');

  // Break-even ratio and DCR
  const yearlyOpEx = operatingExpenses * 12;
  const yearlyDebt = loanPayment * 12;
  const yearlyRent = grossRent * 12;
  const ber = yearlyRent > 0 ? ((yearlyOpEx + yearlyDebt) / yearlyRent) * 100 : 0;
  setText('ber', formatPercent(ber));

  const dcr = yearlyDebt > 0 ? (noi * 12 / yearlyDebt) : 0;
  setText('dcr', dcr.toFixed(2));

  const debtYield = loanAmount > 0 ? ((noi * 12) / loanAmount) * 100 : 0;
  setText('debt-yield', formatPercent(debtYield));

  // Quick stats again w/ proper signs
  setText('stat-cash-flow', formatCurrency(monthlyCashFlow) + '/mo');
  setText('stat-cap-rate', formatPercent(capRatePurchase));
  setText('stat-coc', formatPercent(coc));

  // Draw expenses pie chart
  google.charts.setOnLoadCallback(() => drawExpensesChart({
    taxes, insurance, management, maintenance, capex, hoa, utilities, otherExp
  }));
}

function drawExpensesChart(exp) {
  const data = new google.visualization.DataTable();
  data.addColumn('string', 'Name');
  data.addColumn('number', 'Value');

  data.addRow(['Property Taxes', exp.taxes]);
  data.addRow(['Insurance', exp.insurance]);
  data.addRow(['Property Management', exp.management]);
  data.addRow(['Maintenance', exp.maintenance]);
  data.addRow(['Capital Expenditures', exp.capex]);
  data.addRow(['HOA Fees', exp.hoa]);
  data.addRow(['Utilities', exp.utilities]);
  data.addRow(['Other', exp.otherExp]);

  const options = {
    legend: { position: 'right' },
    pieHole: 0.4,
    chartArea: { width: '80%', height: '80%' }
  };

  const chart = new google.visualization.PieChart(
    document.getElementById('summary-expenses-chart')
  );
  chart.draw(data, options);
}

// Helper functions (you may already have some of these in property-common.js;
// if so, you can remove duplicates and use the shared ones instead)
function setText(id, text) {
  const el = document.getElementById(id);
  if (el) el.textContent = text;
}

function setHtml(id, html) {
  const el = document.getElementById(id);
  if (el) el.innerHTML = html;
}

function toNumber(x) {
  if (x === null || x === undefined || x === '') return 0;
  const n = Number(x);
  return isNaN(n) ? 0 : n;
}

function formatCurrency(value) {
  return '$ ' + Number(value || 0).toLocaleString(undefined, { maximumFractionDigits: 0 });
}

function formatNumber(value) {
  return Number(value || 0).toLocaleString();
}

function formatPercent(value) {
  return Number(value || 0).toFixed(1) + ' %';
}

function escapeHtml(str) {
  if (str === null || str === undefined) return '';
  return String(str)
    .replace(/&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;');
}
