<?php
// listing.php
// Displays details for a single property based on its ID passed via query
// parameter. If the listing does not exist, a friendly error is shown.

require_once __DIR__ . '/data.php';
include __DIR__ . '/header.php';

// Extract ID from the query string
$id = isset($_GET['id']) ? $_GET['id'] : '';

// Find the listing with the given ID
$current = null;
foreach ($listings as $listing) {
    if ($listing['id'] === $id) {
        $current = $listing;
        break;
    }
}

if (!$current): ?>
    <div class="page">
        <h1>Home not found</h1>
        <p>This listing may have been removed or is not available.</p>
    </div>
    <?php include __DIR__ . '/footer.php';
    return;
endif;

// Calculate a simple monthly cost estimate (principal & interest) using
// approximate 0.6% per year of price divided by 10 to approximate tax
// and insurance; this is purely illustrative and not intended to be
// accurate financial advice.
$monthlyEstimate = round(($current['price'] * 0.006) / 10, -1);

// ---------------------------------------------------------------------------
// Additional computed values for enhanced detail sections
//
// The following helper values are computed here so they can be reused in
// several of the detail sections below. They provide rough ranges and
// placeholders for property valuation, comparable sales and rentals, and
// market metrics based off of the basic listing data. These values are
// illustrative only; in a production environment you would fetch real
// analytics from an API such as RentCast.

// AVM (Automated Valuation Model) estimate and range
$valuation            = $current['price'];
$valuationRangeLow    = round($valuation * 0.95);
$valuationRangeHigh   = round($valuation * 1.05);
$valuationConfidence  = 80; // percentage confidence placeholder

// Rent estimate and range based on RentCast mock data if available
$rentEstimate         = isset($current['rentcast']['rentEstimate']) ? $current['rentcast']['rentEstimate'] : null;
$rentRangeLow         = $rentEstimate ? round($rentEstimate * 0.9) : null;
$rentRangeHigh        = $rentEstimate ? round($rentEstimate * 1.1) : null;
$rentConfidence       = $rentEstimate ? 80 : null; // confidence placeholder

// Build a simple array of comparable sales based on the current listing
// Each comp uses a slight variation on the current price and sale date
$comps = [
    [
        'address'       => $current['address'] . ', ' . $current['city'],
        'distance'      => 0.5,
        'bedrooms'      => $current['beds'],
        'bathrooms'     => $current['baths'],
        'squareFootage' => $current['sqft'],
        'yearBuilt'     => 2017,
        'salePrice'     => round($current['price'] * 0.95),
        'saleDate'      => date('Y-m-d', strtotime('-3 months')),
    ],
    [
        'address'       => $current['address'] . ', ' . $current['city'],
        'distance'      => 1.1,
        'bedrooms'      => $current['beds'],
        'bathrooms'     => $current['baths'],
        'squareFootage' => $current['sqft'],
        'yearBuilt'     => 2017,
        'salePrice'     => round($current['price'] * 1.02),
        'saleDate'      => date('Y-m-d', strtotime('-6 months')),
    ],
    [
        'address'       => $current['address'] . ', ' . $current['city'],
        'distance'      => 1.8,
        'bedrooms'      => $current['beds'],
        'bathrooms'     => $current['baths'],
        'squareFootage' => $current['sqft'],
        'yearBuilt'     => 2017,
        'salePrice'     => round($current['price'] * 1.1),
        'saleDate'      => date('Y-m-d', strtotime('-9 months')),
    ],
];

// Build a simple array of comparable rentals based on the current listing
$rentComps = [];
if ($rentEstimate) {
    $rentComps[] = [
        'address'       => $current['address'] . ', ' . $current['city'],
        'distance'      => 0.4,
        'bedrooms'      => $current['beds'],
        'bathrooms'     => $current['baths'],
        'squareFootage' => $current['sqft'],
        'rent'          => round($rentEstimate * 0.9),
        'listedDate'    => date('Y-m-d', strtotime('-2 months')),
    ];
    $rentComps[] = [
        'address'       => $current['address'] . ', ' . $current['city'],
        'distance'      => 1.0,
        'bedrooms'      => $current['beds'],
        'bathrooms'     => $current['baths'],
        'squareFootage' => $current['sqft'],
        'rent'          => round($rentEstimate * 1.0),
        'listedDate'    => date('Y-m-d', strtotime('-5 months')),
    ];
    $rentComps[] = [
        'address'       => $current['address'] . ', ' . $current['city'],
        'distance'      => 1.5,
        'bedrooms'      => $current['beds'],
        'bathrooms'     => $current['baths'],
        'squareFootage' => $current['sqft'],
        'rent'          => round($rentEstimate * 1.1),
        'listedDate'    => date('Y-m-d', strtotime('-8 months')),
    ];
}

// Market metrics placeholders derived from listing data and rentcast metrics
$marketSaleAvgPrice      = round($current['price']);
$marketSaleMedianPrice   = round($current['price'] * 0.98);
$marketPricePerSqFt      = round($current['price'] / $current['sqft']);
$marketSalesVolume       = 50; // number of recent sales in area (placeholder)
$marketSaleCount         = 120; // count of properties sold (placeholder)
$marketDaysOnMarket      = 30;
$marketInventory         = 100;
$marketRentAvg           = $rentEstimate ?: 0;
$marketRentMedian        = $rentEstimate ? round($rentEstimate * 0.95) : 0;
$marketRentPerSqFt       = $rentEstimate ? round($rentEstimate / $current['sqft']) : 0;
$marketRentCount         = 80;
$marketVacancyRate       = isset($current['rentcast']['vacancyRate']) ? $current['rentcast']['vacancyRate'] : 0;

// Simple trend data for three recent months (placeholders)
$saleTrends = [
    ['month' => date('M', strtotime('-2 months')), 'year' => date('Y'), 'medianSalePrice' => round($current['price'] * 0.97), 'averageSalePrice' => round($current['price'] * 1.0)],
    ['month' => date('M', strtotime('-1 months')), 'year' => date('Y'), 'medianSalePrice' => round($current['price'] * 0.98), 'averageSalePrice' => round($current['price'] * 1.01)],
    ['month' => date('M'), 'year' => date('Y'), 'medianSalePrice' => round($current['price']), 'averageSalePrice' => round($current['price'] * 1.02)],
];
$rentTrends = [];
if ($rentEstimate) {
    $rentTrends[] = ['month' => date('M', strtotime('-2 months')), 'year' => date('Y'), 'medianRent' => round($rentEstimate * 0.95), 'averageRent' => round($rentEstimate)];
    $rentTrends[] = ['month' => date('M', strtotime('-1 months')), 'year' => date('Y'), 'medianRent' => round($rentEstimate * 0.97), 'averageRent' => round($rentEstimate * 1.03)];
    $rentTrends[] = ['month' => date('M'), 'year' => date('Y'), 'medianRent' => round($rentEstimate), 'averageRent' => round($rentEstimate * 1.05)];
}

// Additional placeholder photos: replicate the primary image three times for a gallery
$photos = [
    $current['imageUrl'],
    $current['imageUrl'],
    $current['imageUrl'],
];

?>

<div class="listing-detail-page">
    <div class="listing-detail-gallery">
        <img src="<?php echo htmlspecialchars($current['imageUrl']); ?>" alt="<?php echo htmlspecialchars($current['address']); ?>">
    </div>

    <div class="listing-detail-main">
        <div class="listing-detail-info">
            <div class="listing-detail-price-row">
                <h1>$<?php echo number_format($current['price']); ?></h1>
                <span class="listing-detail-meta">
                    <?php echo (int)$current['beds']; ?> bd ·
                    <?php echo (int)$current['baths']; ?> ba ·
                    <?php echo number_format($current['sqft']); ?> sqft
                </span>
            </div>
            <div class="listing-detail-address">
                <?php echo htmlspecialchars($current['address'] . ', ' . $current['city'] . ', ' . $current['state'] . ' ' . $current['zip']); ?>
            </div>

            <section class="detail-section">
                <h2>Overview</h2>
                <p><?php echo htmlspecialchars($current['description']); ?></p>
            </section>

            <section class="detail-section detail-section-grid">
                <div>
                    <h3>Facts &amp; features</h3>
                    <ul class="detail-list">
                        <li>Type: Single family residence</li>
                        <li>Year built: 2017</li>
                        <li>Heating: Central</li>
                        <li>Cooling: Central air</li>
                        <li>Parking: 2‑car garage</li>
                        <li>HOA: $45/month (example)</li>
                    </ul>
                </div>
                <div>
                    <h3>Monthly cost estimate</h3>
                    <p class="detail-highlight">$<?php echo number_format($monthlyEstimate); ?> / month</p>
                    <ul class="detail-list small">
                        <li>Principal &amp; interest</li>
                        <li>Property taxes</li>
                        <li>Home insurance</li>
                        <li>HOA fees (if applicable)</li>
                    </ul>
                    <button class="primary-btn full" type="button" onclick="window.location.href='/zillow/z0.0/mortgage.php'">Get pre‑qualified</button>
                </div>
            </section>

            <?php if (isset($current['rentcast']) && is_array($current['rentcast'])): ?>
            <!-- RentCast data section. These values are placeholders provided in data.php
                 and demonstrate how additional rental insights from the RentCast API
                 could be displayed. Replace or augment this section with live API
                 values as needed. -->
            <section class="detail-section">
                <h2>Rental insights</h2>
                <p>Key metrics based on RentCast data for this property and its surrounding market.</p>
                <ul class="detail-list">
                    <?php if (isset($current['rentcast']['rentEstimate'])): ?>
                    <li>Estimated monthly rent: $<?php echo number_format($current['rentcast']['rentEstimate']); ?></li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['avgRent1br'])): ?>
                    <li>Average 1BR rent in area: $<?php echo number_format($current['rentcast']['avgRent1br']); ?></li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['avgRent2br'])): ?>
                    <li>Average 2BR rent in area: $<?php echo number_format($current['rentcast']['avgRent2br']); ?></li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['avgRent3br'])): ?>
                    <li>Average 3BR rent in area: $<?php echo number_format($current['rentcast']['avgRent3br']); ?></li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['avgRent4br'])): ?>
                    <li>Average 4BR rent in area: $<?php echo number_format($current['rentcast']['avgRent4br']); ?></li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['vacancyRate'])): ?>
                    <li>Vacancy rate: <?php echo number_format($current['rentcast']['vacancyRate'], 1); ?>%</li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['capRate'])): ?>
                    <li>Cap rate: <?php echo number_format($current['rentcast']['capRate'], 1); ?>%</li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['rentalGrowthYoy'])): ?>
                    <li>Year-over-year rental growth: <?php echo number_format($current['rentcast']['rentalGrowthYoy'], 1); ?>%</li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['medianDaysOnMarket'])): ?>
                    <li>Median days on market: <?php echo (int)$current['rentcast']['medianDaysOnMarket']; ?> days</li>
                    <?php endif; ?>
                </ul>
            </section>
            <?php endif; ?>

            <!-- Property details section -->
            <section class="detail-section">
                <h2>Property details</h2>
                <div class="detail-grid">
                    <div>
                        <h3>Address &amp; location</h3>
                        <ul class="detail-list">
                            <li>Formatted address: <?php echo htmlspecialchars($current['address'] . ', ' . $current['city'] . ', ' . $current['state'] . ' ' . $current['zip']); ?></li>
                            <li>Property ID: <?php echo htmlspecialchars($current['id']); ?></li>
                            <li>County: <?php echo htmlspecialchars($current['city']); ?> County</li>
                            <li>Latitude / Longitude: N/A</li>
                        </ul>
                    </div>
                    <div>
                        <h3>Structure &amp; specs</h3>
                        <ul class="detail-list">
                            <li>Property type: Single family residence</li>
                            <li>Bedrooms: <?php echo (int)$current['beds']; ?> • Bathrooms: <?php echo (int)$current['baths']; ?></li>
                            <li>Square footage: <?php echo number_format($current['sqft']); ?> sqft</li>
                            <li>Lot size: N/A</li>
                            <li>Year built: 2017</li>
                        </ul>
                    </div>
                    <div>
                        <h3>Ownership &amp; taxes</h3>
                        <ul class="detail-list">
                            <li>Owner occupied: N/A</li>
                            <li>Tax assessment: N/A</li>
                            <li>HOA: $45/month (example)</li>
                            <li>Last sale date: <?php echo date('M d, Y', strtotime('-1 years')); ?></li>
                        </ul>
                    </div>
                </div>
            </section>

            <!-- Value estimate (AVM) section -->
            <section class="detail-section">
                <h2>Value estimate (AVM)</h2>
                <p>Estimated value based on recent sales of similar properties.</p>
                <p class="detail-highlight">$<?php echo number_format($valuation); ?> <span style="font-size:14px; color:var(--text-muted);">(Confidence <?php echo $valuationConfidence; ?>%)</span></p>
                <p>Range: $<?php echo number_format($valuationRangeLow); ?> – $<?php echo number_format($valuationRangeHigh); ?></p>
                <h3>Comparable sales</h3>
                <ul class="detail-list">
                    <?php foreach ($comps as $comp): ?>
                    <li>
                        <strong><?php echo htmlspecialchars($comp['address']); ?></strong><br>
                        <?php echo (int)$comp['bedrooms']; ?> bd • <?php echo (int)$comp['bathrooms']; ?> ba • <?php echo number_format($comp['squareFootage']); ?> sqft<br>
                        Sold for $<?php echo number_format($comp['salePrice']); ?> on <?php echo date('M d, Y', strtotime($comp['saleDate'])); ?> (<?php echo number_format($comp['distance'], 1); ?> mi away)
                    </li>
                    <?php endforeach; ?>
                </ul>
            </section>

            <!-- Rent estimate & comparable rentals section -->
            <?php if ($rentEstimate): ?>
            <section class="detail-section">
                <h2>Rent estimate &amp; rentals</h2>
                <p>Estimated long‑term rent based on comparable rentals in the area.</p>
                <p class="detail-highlight">$<?php echo number_format($rentEstimate); ?> <span style="font-size:14px; color:var(--text-muted);">(Confidence <?php echo $rentConfidence; ?>%)</span></p>
                <p>Range: $<?php echo number_format($rentRangeLow); ?> – $<?php echo number_format($rentRangeHigh); ?></p>
                <h3>Comparable rentals</h3>
                <ul class="detail-list">
                    <?php foreach ($rentComps as $rcomp): ?>
                    <li>
                        <strong><?php echo htmlspecialchars($rcomp['address']); ?></strong><br>
                        <?php echo (int)$rcomp['bedrooms']; ?> bd • <?php echo (int)$rcomp['bathrooms']; ?> ba • <?php echo number_format($rcomp['squareFootage']); ?> sqft<br>
                        Rents for $<?php echo number_format($rcomp['rent']); ?> (listed on <?php echo date('M d, Y', strtotime($rcomp['listedDate'])); ?>, <?php echo number_format($rcomp['distance'], 1); ?> mi away)
                    </li>
                    <?php endforeach; ?>
                </ul>
                <h3>Market metrics</h3>
                <ul class="detail-list">
                    <li>Average rent: $<?php echo number_format($marketRentAvg); ?> • Median rent: $<?php echo number_format($marketRentMedian); ?></li>
                    <li>Rent per sqft: $<?php echo number_format($marketRentPerSqFt); ?></li>
                    <li>Rental count: <?php echo (int)$marketRentCount; ?></li>
                    <li>Vacancy rate: <?php echo number_format($marketVacancyRate, 1); ?>%</li>
                </ul>
            </section>
            <?php endif; ?>

            <!-- Active sale listing details section -->
            <section class="detail-section">
                <h2>Sale listing details</h2>
                <p>Current listing information for this property.</p>
                <ul class="detail-list">
                    <li>Listing ID: <?php echo strtoupper($current['id']) . '-LIST'; ?></li>
                    <li>Status: For sale</li>
                    <li>List price: $<?php echo number_format($current['price']); ?></li>
                    <li>Price per sqft: $<?php echo number_format($marketPricePerSqFt); ?></li>
                    <li>Days on market: <?php echo (int)$marketDaysOnMarket; ?></li>
                    <li>List date: <?php echo date('M d, Y', strtotime('-1 months')); ?></li>
                    <li>Last update: <?php echo date('M d, Y'); ?></li>
                </ul>
                <h3>Listing agent</h3>
                <ul class="detail-list">
                    <li>Agent name: Jane Agent</li>
                    <li>Phone: (555) 123‑4567</li>
                    <li>Email: jane.agent@example.com</li>
                    <li>Office: Home9 Realty</li>
                </ul>
                <h3>Open house</h3>
                <ul class="detail-list">
                    <li><?php echo date('M d, Y', strtotime('+5 days')); ?>, 1:00&nbsp;–&nbsp;3:00&nbsp;PM</li>
                </ul>
            </section>

            <!-- Rental listing details section -->
            <?php if ($rentEstimate): ?>
            <section class="detail-section">
                <h2>Rental listing details</h2>
                <p>Information for the long‑term rental version of this property.</p>
                <ul class="detail-list">
                    <li>Listing ID: <?php echo strtoupper($current['id']) . '-RENT'; ?></li>
                    <li>Status: Available for rent</li>
                    <li>Rent: $<?php echo number_format($rentEstimate); ?></li>
                    <li>Deposit: $<?php echo number_format($rentEstimate); ?></li>
                    <li>Application fee: $45</li>
                    <li>Lease terms: 12 months</li>
                    <li>Available from: <?php echo date('M d, Y', strtotime('+1 weeks')); ?></li>
                    <li>Days on market: <?php echo (int)$marketDaysOnMarket; ?></li>
                </ul>
            </section>
            <?php endif; ?>

            <!-- Market data & trends section -->
            <section class="detail-section">
                <h2>Market data &amp; trends</h2>
                <div class="detail-grid">
                    <div>
                        <h3>Sale market</h3>
                        <ul class="detail-list">
                            <li>Average sale price: $<?php echo number_format($marketSaleAvgPrice); ?></li>
                            <li>Median sale price: $<?php echo number_format($marketSaleMedianPrice); ?></li>
                            <li>Average price per sqft: $<?php echo number_format($marketPricePerSqFt); ?></li>
                            <li>Sales volume: <?php echo (int)$marketSalesVolume; ?></li>
                            <li>Sales count: <?php echo (int)$marketSaleCount; ?></li>
                            <li>Days on market: <?php echo (int)$marketDaysOnMarket; ?></li>
                            <li>Inventory: <?php echo (int)$marketInventory; ?></li>
                        </ul>
                    </div>
                    <div>
                        <h3>Rental market</h3>
                        <ul class="detail-list">
                            <li>Average rent: $<?php echo number_format($marketRentAvg); ?></li>
                            <li>Median rent: $<?php echo number_format($marketRentMedian); ?></li>
                            <li>Rent per sqft: $<?php echo number_format($marketRentPerSqFt); ?></li>
                            <li>Rental count: <?php echo (int)$marketRentCount; ?></li>
                            <li>Vacancy rate: <?php echo number_format($marketVacancyRate, 1); ?>%</li>
                        </ul>
                    </div>
                </div>
                <h3>Sale trends (last 3 months)</h3>
                <ul class="detail-list small">
                    <?php foreach ($saleTrends as $trend): ?>
                    <li><?php echo $trend['month'] . ' ' . $trend['year']; ?> – Median: $<?php echo number_format($trend['medianSalePrice']); ?>; Average: $<?php echo number_format($trend['averageSalePrice']); ?></li>
                    <?php endforeach; ?>
                </ul>
                <?php if (!empty($rentTrends)): ?>
                <h3>Rent trends (last 3 months)</h3>
                <ul class="detail-list small">
                    <?php foreach ($rentTrends as $trend): ?>
                    <li><?php echo $trend['month'] . ' ' . $trend['year']; ?> – Median: $<?php echo number_format($trend['medianRent']); ?>; Average: $<?php echo number_format($trend['averageRent']); ?></li>
                    <?php endforeach; ?>
                </ul>
                <?php endif; ?>
            </section>

            <!-- Property photos section -->
            <section class="detail-section">
                <h2>Property photos</h2>
                <div class="photo-grid">
                    <?php foreach ($photos as $url): ?>
                    <img src="<?php echo htmlspecialchars($url); ?>" alt="Property photo">
                    <?php endforeach; ?>
                </div>
            </section>

            <section class="detail-section">
                <h2>Neighborhood</h2>
                <p>Show commute times, nearby schools, and neighborhood statistics here. This section can be powered by your own data APIs.</p>
            </section>
        </div>

        <aside class="listing-detail-sidebar">
            <div class="detail-card">
                <h3>Contact an agent</h3>
                <p>Connect with a local professional to learn more about this home.</p>
                <form class="detail-form" onsubmit="event.preventDefault(); alert('Thank you! An agent will reach out.');">
                    <input type="text" placeholder="Name" required>
                    <input type="email" placeholder="Email" required>
                    <input type="tel" placeholder="Phone" required>
                    <textarea placeholder="I'm interested in this home."></textarea>
                    <button class="primary-btn full" type="submit">Request info</button>
                </form>
            </div>

            <div class="detail-card">
                <h3>Tour this home</h3>
                <p>Select a date to schedule an in‑person or virtual tour.</p>
                <input type="date" style="width:100%; margin-bottom:8px; padding:8px; border-radius:10px; border:1px solid var(--border-subtle);">
                <button class="secondary-btn full" type="button" onclick="alert('Your tour request has been submitted!')">Schedule a tour</button>
            </div>
        </aside>
    </div>
</div>

<?php
include __DIR__ . '/footer.php';