// script.js
// Global JavaScript used across the site. Handles toggling the user
// avatar menu and performing mortgage calculations on the fly.

/**
 * Toggle visibility of the avatar dropdown menu. Bound to the avatar button
 * onclick in header.php. Clicking elsewhere on the document hides the menu.
 */
function toggleAvatarMenu() {
    const dropdown = document.getElementById('avatarDropdown');
    if (!dropdown) return;
    const current = dropdown.style.display;
    dropdown.style.display = current === 'none' || current === '' ? 'block' : 'none';
}

// Close avatar menu when clicking outside
document.addEventListener('click', function (event) {
    const menu = document.getElementById('avatarMenu');
    const dropdown = document.getElementById('avatarDropdown');
    if (!menu || !dropdown) return;
    if (!menu.contains(event.target)) {
        dropdown.style.display = 'none';
    }
});

/**
 * Mortgage calculator: updates the monthly payment display when any input
 * changes. Expects inputs with ids homePrice, downPayment, rate, term and
 * an element with id monthlyPayment to display the result.
 */
function initMortgageCalculator() {
    const homePriceInput = document.getElementById('homePrice');
    const downPaymentInput = document.getElementById('downPayment');
    const rateInput = document.getElementById('rate');
    const termInput = document.getElementById('term');
    const monthlyPaymentEl = document.getElementById('monthlyPayment');
    if (!homePriceInput || !downPaymentInput || !rateInput || !termInput || !monthlyPaymentEl) {
        return;
    }
    function updatePayment() {
        const homePrice = parseFloat(homePriceInput.value) || 0;
        const downPayment = parseFloat(downPaymentInput.value) || 0;
        const rate = parseFloat(rateInput.value) || 0;
        const term = parseFloat(termInput.value) || 0;
        const principal = homePrice * (1 - downPayment / 100);
        const monthlyRate = rate / 100 / 12;
        const n = term * 12;
        let payment;
        if (monthlyRate === 0) {
            payment = n === 0 ? 0 : principal / n;
        } else {
            payment = (principal * monthlyRate) / (1 - Math.pow(1 + monthlyRate, -n));
        }
        monthlyPaymentEl.textContent = isFinite(payment) ? `$${Math.round(payment).toLocaleString()} / month` : '$0 / month';
    }
    homePriceInput.addEventListener('input', updatePayment);
    downPaymentInput.addEventListener('input', updatePayment);
    rateInput.addEventListener('input', updatePayment);
    termInput.addEventListener('input', updatePayment);
    updatePayment();
}

// Initialize mortgage calculator on DOMContentLoaded
document.addEventListener('DOMContentLoaded', initMortgageCalculator);