<?php
// listing.php
// Displays details for a single property based on its ID passed via query
// parameter. If the listing does not exist, a friendly error is shown.

require_once __DIR__ . '/data.php';
include __DIR__ . '/header.php';

// Extract ID from the query string
$id = isset($_GET['id']) ? $_GET['id'] : '';

// Find the listing with the given ID
$current = null;
foreach ($listings as $listing) {
    if ($listing['id'] === $id) {
        $current = $listing;
        break;
    }
}

if (!$current): ?>
    <div class="page">
        <h1>Home not found</h1>
        <p>This listing may have been removed or is not available.</p>
    </div>
    <?php include __DIR__ . '/footer.php';
    return;
endif;

// Calculate a simple monthly cost estimate (principal & interest) using
// approximate 0.6% per year of price divided by 10 to approximate tax
// and insurance; this is purely illustrative and not intended to be
// accurate financial advice.
$monthlyEstimate = round(($current['price'] * 0.006) / 10, -1);

// ---------------------------------------------------------------------------
// Additional computed values for enhanced detail sections
//
// The following helper values are computed here so they can be reused in
// several of the detail sections below. They provide rough ranges and
// placeholders for property valuation, comparable sales and rentals, and
// market metrics based off of the basic listing data. These values are
// illustrative only; in a production environment you would fetch real
// analytics from an API such as RentCast.

// AVM (Automated Valuation Model) estimate and range
$valuation            = $current['price'];
$valuationRangeLow    = round($valuation * 0.95);
$valuationRangeHigh   = round($valuation * 1.05);
$valuationConfidence  = 80; // percentage confidence placeholder

// Rent estimate and range based on RentCast mock data if available
$rentEstimate         = isset($current['rentcast']['rentEstimate']) ? $current['rentcast']['rentEstimate'] : null;
$rentRangeLow         = $rentEstimate ? round($rentEstimate * 0.9) : null;
$rentRangeHigh        = $rentEstimate ? round($rentEstimate * 1.1) : null;
$rentConfidence       = $rentEstimate ? 80 : null; // confidence placeholder

// Build a simple array of comparable sales based on the current listing
// Each comp uses a slight variation on the current price and sale date
$comps = [
    [
        'address'       => $current['address'] . ', ' . $current['city'],
        'distance'      => 0.5,
        'bedrooms'      => $current['beds'],
        'bathrooms'     => $current['baths'],
        'squareFootage' => $current['sqft'],
        'yearBuilt'     => 2017,
        'salePrice'     => round($current['price'] * 0.95),
        'saleDate'      => date('Y-m-d', strtotime('-3 months')),
    ],
    [
        'address'       => $current['address'] . ', ' . $current['city'],
        'distance'      => 1.1,
        'bedrooms'      => $current['beds'],
        'bathrooms'     => $current['baths'],
        'squareFootage' => $current['sqft'],
        'yearBuilt'     => 2017,
        'salePrice'     => round($current['price'] * 1.02),
        'saleDate'      => date('Y-m-d', strtotime('-6 months')),
    ],
    [
        'address'       => $current['address'] . ', ' . $current['city'],
        'distance'      => 1.8,
        'bedrooms'      => $current['beds'],
        'bathrooms'     => $current['baths'],
        'squareFootage' => $current['sqft'],
        'yearBuilt'     => 2017,
        'salePrice'     => round($current['price'] * 1.1),
        'saleDate'      => date('Y-m-d', strtotime('-9 months')),
    ],
];

// Build a simple array of comparable rentals based on the current listing
$rentComps = [];
if ($rentEstimate) {
    $rentComps[] = [
        'address'       => $current['address'] . ', ' . $current['city'],
        'distance'      => 0.4,
        'bedrooms'      => $current['beds'],
        'bathrooms'     => $current['baths'],
        'squareFootage' => $current['sqft'],
        'rent'          => round($rentEstimate * 0.9),
        'listedDate'    => date('Y-m-d', strtotime('-2 months')),
    ];
    $rentComps[] = [
        'address'       => $current['address'] . ', ' . $current['city'],
        'distance'      => 1.0,
        'bedrooms'      => $current['beds'],
        'bathrooms'     => $current['baths'],
        'squareFootage' => $current['sqft'],
        'rent'          => round($rentEstimate * 1.0),
        'listedDate'    => date('Y-m-d', strtotime('-5 months')),
    ];
    $rentComps[] = [
        'address'       => $current['address'] . ', ' . $current['city'],
        'distance'      => 1.5,
        'bedrooms'      => $current['beds'],
        'bathrooms'     => $current['baths'],
        'squareFootage' => $current['sqft'],
        'rent'          => round($rentEstimate * 1.1),
        'listedDate'    => date('Y-m-d', strtotime('-8 months')),
    ];
}

// Prepare labels and values for comparable sales and rentals (for bar charts).
// Comparable bar chart arrays are no longer used because comparables are shown as tables.
// $compLabels = array_map(function ($c) {
//     return $c['address'];
// }, $comps);
// $compSalePrices = array_map(function ($c) {
//     return $c['salePrice'];
// }, $comps);
// $rentCompLabels = array_map(function ($c) {
//     return $c['address'];
// }, $rentComps);
// $rentCompPrices = array_map(function ($c) {
//     return $c['rent'];
// }, $rentComps);

// Market metrics placeholders derived from listing data and rentcast metrics
$marketSaleAvgPrice      = round($current['price']);
$marketSaleMedianPrice   = round($current['price'] * 0.98);
$marketPricePerSqFt      = round($current['price'] / $current['sqft']);
$marketSalesVolume       = 50; // number of recent sales in area (placeholder)
$marketSaleCount         = 120; // count of properties sold (placeholder)
$marketDaysOnMarket      = 30;
$marketInventory         = 100;
$marketRentAvg           = $rentEstimate ?: 0;
$marketRentMedian        = $rentEstimate ? round($rentEstimate * 0.95) : 0;
$marketRentPerSqFt       = $rentEstimate ? round($rentEstimate / $current['sqft']) : 0;
$marketRentCount         = 80;
$marketVacancyRate       = isset($current['rentcast']['vacancyRate']) ? $current['rentcast']['vacancyRate'] : 0;

// Simple trend data for three recent months (placeholders)
$saleTrends = [
    ['month' => date('M', strtotime('-2 months')), 'year' => date('Y'), 'medianSalePrice' => round($current['price'] * 0.97), 'averageSalePrice' => round($current['price'] * 1.0)],
    ['month' => date('M', strtotime('-1 months')), 'year' => date('Y'), 'medianSalePrice' => round($current['price'] * 0.98), 'averageSalePrice' => round($current['price'] * 1.01)],
    ['month' => date('M'), 'year' => date('Y'), 'medianSalePrice' => round($current['price']), 'averageSalePrice' => round($current['price'] * 1.02)],
];
$rentTrends = [];
if ($rentEstimate) {
    $rentTrends[] = ['month' => date('M', strtotime('-2 months')), 'year' => date('Y'), 'medianRent' => round($rentEstimate * 0.95), 'averageRent' => round($rentEstimate)];
    $rentTrends[] = ['month' => date('M', strtotime('-1 months')), 'year' => date('Y'), 'medianRent' => round($rentEstimate * 0.97), 'averageRent' => round($rentEstimate * 1.03)];
    $rentTrends[] = ['month' => date('M'), 'year' => date('Y'), 'medianRent' => round($rentEstimate), 'averageRent' => round($rentEstimate * 1.05)];
}

// Additional placeholder photos: replicate the primary image three times for a gallery
$photos = [
    $current['imageUrl'],
    $current['imageUrl'],
    $current['imageUrl'],
];

?>

<div class="listing-detail-page">
    <div class="listing-detail-gallery">
        <img src="<?php echo htmlspecialchars($current['imageUrl']); ?>" alt="<?php echo htmlspecialchars($current['address']); ?>">
    </div>

    <!-- Property photos moved underneath the main banner image -->
    <section class="detail-section">
        <h2>Property photos</h2>
        <div class="photo-grid">
            <?php foreach ($photos as $url): ?>
            <img src="<?php echo htmlspecialchars($url); ?>" alt="Property photo">
            <?php endforeach; ?>
        </div>
    </section>

    <div class="listing-detail-main">
        <div class="listing-detail-info">
            <div class="listing-detail-price-row">
                <h1>$<?php echo number_format($current['price']); ?></h1>
                <span class="listing-detail-meta">
                    <?php echo (int)$current['beds']; ?> bd ·
                    <?php echo (int)$current['baths']; ?> ba ·
                    <?php echo number_format($current['sqft']); ?> sqft
                </span>
            </div>
            <div class="listing-detail-address">
                <?php echo htmlspecialchars($current['address'] . ', ' . $current['city'] . ', ' . $current['state'] . ' ' . $current['zip']); ?>
            </div>

            <section class="detail-section">
                <h2>Overview</h2>
                <p><?php echo htmlspecialchars($current['description']); ?></p>
            </section>

            <section class="detail-section detail-section-grid">
                <div>
                    <h3>Facts &amp; features</h3>
                    <ul class="detail-list">
                        <li>Type: Single family residence</li>
                        <li>Year built: 2017</li>
                        <li>Heating: Central</li>
                        <li>Cooling: Central air</li>
                        <li>Parking: 2‑car garage</li>
                        <li>HOA: $45/month (example)</li>
                    </ul>
                </div>
                <div>
                    <h3>Monthly cost estimate</h3>
                    <p class="detail-highlight">$<?php echo number_format($monthlyEstimate); ?> / month</p>
                    <ul class="detail-list small">
                        <li>Principal &amp; interest</li>
                        <li>Property taxes</li>
                        <li>Home insurance</li>
                        <li>HOA fees (if applicable)</li>
                    </ul>
                    <button class="primary-btn full" type="button" onclick="window.location.href='/zillow/z0.0/mortgage.php'">Get pre‑qualified</button>
                </div>
                <!-- Inserted structure & specs beneath facts & features -->
                <div>
                    <h3>Structure &amp; specs</h3>
                    <ul class="detail-list">
                        <li>Property type: Single family residence</li>
                        <li>Bedrooms: <?php echo (int)$current['beds']; ?> &bullet; Bathrooms: <?php echo (int)$current['baths']; ?></li>
                        <li>Square footage: <?php echo number_format($current['sqft']); ?> sqft</li>
                        <li>Lot size: N/A</li>
                        <li>Year built: 2017</li>
                    </ul>
                </div>
            </section>
            <!-- Inserted location section with map and details -->
            <section class="detail-section">
                <h2>Location</h2>
                <div class="map-container">
                    <iframe
                        width="100%"
                        height="300"
                        frameborder="0"
                        style="border:0; border-radius: var(--radius-lg); box-shadow: var(--shadow-soft);"
                        src="https://www.google.com/maps?q=<?php echo urlencode($current['address'] . ' ' . $current['city'] . ' ' . $current['state'] . ' ' . $current['zip']); ?>&output=embed"
                        allowfullscreen>
                    </iframe>
                    <div class="location-details">
                        <p><strong>Formatted address:</strong> <?php echo htmlspecialchars($current['address'] . ', ' . $current['city'] . ', ' . $current['state'] . ' ' . $current['zip']); ?></p>
                        <p><strong>Property ID:</strong> <?php echo htmlspecialchars($current['id']); ?></p>
                        <p><strong>County:</strong> <?php echo htmlspecialchars($current['city']); ?> County</p>
                        <p><strong>Coordinates:</strong> N/A</p>
                    </div>
                </div>
            </section>

            <?php if (false && isset($current['rentcast']) && is_array($current['rentcast'])): ?>
            <!-- RentCast data section. These values are placeholders provided in data.php
                 and demonstrate how additional rental insights from the RentCast API
                 could be displayed. Replace or augment this section with live API
                 values as needed. -->
            <section class="detail-section">
                <h2>Rental insights</h2>
                <p>Key metrics based on RentCast data for this property and its surrounding market.</p>
                <ul class="detail-list">
                    <?php if (isset($current['rentcast']['rentEstimate'])): ?>
                    <li>Estimated monthly rent: $<?php echo number_format($current['rentcast']['rentEstimate']); ?></li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['avgRent1br'])): ?>
                    <li>Average 1BR rent in area: $<?php echo number_format($current['rentcast']['avgRent1br']); ?></li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['avgRent2br'])): ?>
                    <li>Average 2BR rent in area: $<?php echo number_format($current['rentcast']['avgRent2br']); ?></li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['avgRent3br'])): ?>
                    <li>Average 3BR rent in area: $<?php echo number_format($current['rentcast']['avgRent3br']); ?></li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['avgRent4br'])): ?>
                    <li>Average 4BR rent in area: $<?php echo number_format($current['rentcast']['avgRent4br']); ?></li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['vacancyRate'])): ?>
                    <li>Vacancy rate: <?php echo number_format($current['rentcast']['vacancyRate'], 1); ?>%</li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['capRate'])): ?>
                    <li>Cap rate: <?php echo number_format($current['rentcast']['capRate'], 1); ?>%</li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['rentalGrowthYoy'])): ?>
                    <li>Year-over-year rental growth: <?php echo number_format($current['rentcast']['rentalGrowthYoy'], 1); ?>%</li>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['medianDaysOnMarket'])): ?>
                    <li>Median days on market: <?php echo (int)$current['rentcast']['medianDaysOnMarket']; ?> days</li>
                    <?php endif; ?>
                </ul>
            </section>
            <?php endif; ?>

            <!-- Property details section -->
            <section class="detail-section">
                <h2>Property details</h2>
                <div class="card-grid">
                    <div class="info-card">
                        <h3>Ownership &amp; taxes</h3>
                        <ul class="detail-list">
                            <li>Owner occupied: N/A</li>
                            <li>Tax assessment: N/A</li>
                            <li>HOA: $45/month (example)</li>
                            <li>Last sale date: <?php echo date('M d, Y', strtotime('-1 years')); ?></li>
                        </ul>
                    </div>
                </div>
            </section>

            <!-- Value estimate (AVM) section -->
            <section class="detail-section">
                <h2>Value estimate (AVM)</h2>
                <p>Estimated value based on recent sales of similar properties.</p>
                <p class="detail-highlight">$<?php echo number_format($valuation); ?> <span style="font-size:14px; color:var(--text-muted);">(Confidence <?php echo $valuationConfidence; ?>%)</span></p>
                <p>Range: $<?php echo number_format($valuationRangeLow); ?> – $<?php echo number_format($valuationRangeHigh); ?></p>
                <h3>Comparable sales</h3>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Address</th>
                            <th>Beds/Baths</th>
                            <th>Size</th>
                            <th>Sale price</th>
                            <th>Sale date</th>
                            <th>Distance</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($comps as $comp): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($comp['address']); ?></td>
                            <td><?php echo (int)$comp['bedrooms']; ?> bd / <?php echo (int)$comp['bathrooms']; ?> ba</td>
                            <td><?php echo number_format($comp['squareFootage']); ?> sqft</td>
                            <td>$<?php echo number_format($comp['salePrice']); ?></td>
                            <td><?php echo date('M d, Y', strtotime($comp['saleDate'])); ?></td>
                            <td><?php echo number_format($comp['distance'], 1); ?> mi</td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </section>

            <?php if (isset($current['rentcast']) && is_array($current['rentcast'])): ?>
            <!-- Rental insights moved here to appear above comparable rentals -->
            <section class="detail-section">
                <h2>Rental insights</h2>
                <p>A snapshot of rental market performance for this property and its neighbourhood.</p>
                <div class="rental-insight-grid">
                    <?php if (isset($current['rentcast']['rentEstimate'])): ?>
                    <div class="insight-card">
                        <span class="insight-label">Monthly rent</span>
                        <span class="insight-value">$<?php echo number_format($current['rentcast']['rentEstimate']); ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['avgRent1br'])): ?>
                    <div class="insight-card">
                        <span class="insight-label">Avg. 1BR rent</span>
                        <span class="insight-value">$<?php echo number_format($current['rentcast']['avgRent1br']); ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['avgRent2br'])): ?>
                    <div class="insight-card">
                        <span class="insight-label">Avg. 2BR rent</span>
                        <span class="insight-value">$<?php echo number_format($current['rentcast']['avgRent2br']); ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['avgRent3br'])): ?>
                    <div class="insight-card">
                        <span class="insight-label">Avg. 3BR rent</span>
                        <span class="insight-value">$<?php echo number_format($current['rentcast']['avgRent3br']); ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['avgRent4br'])): ?>
                    <div class="insight-card">
                        <span class="insight-label">Avg. 4BR rent</span>
                        <span class="insight-value">$<?php echo number_format($current['rentcast']['avgRent4br']); ?></span>
                    </div>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['vacancyRate'])): ?>
                    <div class="insight-card">
                        <span class="insight-label">Vacancy rate</span>
                        <span class="insight-value"><?php echo number_format($current['rentcast']['vacancyRate'], 1); ?>%</span>
                    </div>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['capRate'])): ?>
                    <div class="insight-card">
                        <span class="insight-label">Cap rate</span>
                        <span class="insight-value"><?php echo number_format($current['rentcast']['capRate'], 1); ?>%</span>
                    </div>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['rentalGrowthYoy'])): ?>
                    <div class="insight-card">
                        <span class="insight-label">Rent growth (YoY)</span>
                        <span class="insight-value"><?php echo number_format($current['rentcast']['rentalGrowthYoy'], 1); ?>%</span>
                    </div>
                    <?php endif; ?>
                    <?php if (isset($current['rentcast']['medianDaysOnMarket'])): ?>
                    <div class="insight-card">
                        <span class="insight-label">Median DOM</span>
                        <span class="insight-value"><?php echo (int)$current['rentcast']['medianDaysOnMarket']; ?> days</span>
                    </div>
                    <?php endif; ?>
                </div>
            </section>
            <?php endif; ?>

            <!-- Rent estimate & comparable rentals section -->
            <?php if ($rentEstimate): ?>
            <section class="detail-section">
                <h2>Rent estimate &amp; rentals</h2>
                <p>Estimated long‑term rent based on comparable rentals in the area.</p>
                <p class="detail-highlight">$<?php echo number_format($rentEstimate); ?> <span style="font-size:14px; color:var(--text-muted);">(Confidence <?php echo $rentConfidence; ?>%)</span></p>
                <p>Range: $<?php echo number_format($rentRangeLow); ?> – $<?php echo number_format($rentRangeHigh); ?></p>
                <h3>Comparable rentals</h3>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Address</th>
                            <th>Beds/Baths</th>
                            <th>Size</th>
                            <th>Rent</th>
                            <th>Listed date</th>
                            <th>Distance</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($rentComps as $rcomp): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($rcomp['address']); ?></td>
                            <td><?php echo (int)$rcomp['bedrooms']; ?> bd / <?php echo (int)$rcomp['bathrooms']; ?> ba</td>
                            <td><?php echo number_format($rcomp['squareFootage']); ?> sqft</td>
                            <td>$<?php echo number_format($rcomp['rent']); ?></td>
                            <td><?php echo date('M d, Y', strtotime($rcomp['listedDate'])); ?></td>
                            <td><?php echo number_format($rcomp['distance'], 1); ?> mi</td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
                <h3>Market metrics</h3>
                <ul class="detail-list">
                    <li>Average rent: $<?php echo number_format($marketRentAvg); ?> • Median rent: $<?php echo number_format($marketRentMedian); ?></li>
                    <li>Rent per sqft: $<?php echo number_format($marketRentPerSqFt); ?></li>
                    <li>Rental count: <?php echo (int)$marketRentCount; ?></li>
                    <li>Vacancy rate: <?php echo number_format($marketVacancyRate, 1); ?>%</li>
                </ul>
                <h3>Adjust rent estimate</h3>
                <div class="rent-slider-section">
                    <label for="rentSlider">Use the slider to adjust the rent estimate:</label><br>
                    <input type="range"
                           id="rentSlider"
                           min="<?php echo (int)$rentRangeLow; ?>"
                           max="<?php echo (int)$rentRangeHigh; ?>"
                           value="<?php echo (int)$rentEstimate; ?>"
                           step="50">
                    <p>Adjusted rent: <span id="rentValue">$<?php echo number_format($rentEstimate); ?></span> / month</p>
                </div>
            </section>
            <?php endif; ?>

            <!-- Additional detail sections have been moved to the sidebar below -->

            <!-- Property photos are moved beneath the main image -->

            <section class="detail-section">
                <h2>Neighborhood</h2>
                <p>Show commute times, nearby schools, and neighborhood statistics here. This section can be powered by your own data APIs.</p>
            </section>
        </div>

        <aside class="listing-detail-sidebar">
            <div class="detail-card">
                <h3>Contact an agent</h3>
                <p>Connect with a local professional to learn more about this home.</p>
                <form class="detail-form" onsubmit="event.preventDefault(); alert('Thank you! An agent will reach out.');">
                    <input type="text" placeholder="Name" required>
                    <input type="email" placeholder="Email" required>
                    <input type="tel" placeholder="Phone" required>
                    <textarea placeholder="I'm interested in this home."></textarea>
                    <button class="primary-btn full" type="submit">Request info</button>
                </form>
            </div>

            <div class="detail-card">
                <h3>Tour this home</h3>
                <p>Select a date to schedule an in‑person or virtual tour.</p>
                <input type="date" style="width:100%; margin-bottom:8px; padding:8px; border-radius:10px; border:1px solid var(--border-subtle);">
                <button class="secondary-btn full" type="button" onclick="alert('Your tour request has been submitted!')">Schedule a tour</button>
            </div>

            <!-- Additional less‑important data blocks moved to the right column -->
            <div class="detail-card">
                <h3>Sale listing details</h3>
                <p>Current listing information for this property.</p>
                <ul class="detail-list">
                    <li>Listing ID: <?php echo strtoupper($current['id']) . '-LIST'; ?></li>
                    <li>Status: For sale</li>
                    <li>List price: $<?php echo number_format($current['price']); ?></li>
                    <li>Price per sqft: $<?php echo number_format($marketPricePerSqFt); ?></li>
                    <li>Days on market: <?php echo (int)$marketDaysOnMarket; ?></li>
                    <li>List date: <?php echo date('M d, Y', strtotime('-1 months')); ?></li>
                    <li>Last update: <?php echo date('M d, Y'); ?></li>
                </ul>
                <h4>Listing agent</h4>
                <ul class="detail-list">
                    <li>Agent name: Jane Agent</li>
                    <li>Phone: (555) 123‑4567</li>
                    <li>Email: jane.agent@example.com</li>
                    <li>Office: Home9 Realty</li>
                </ul>
                <h4>Open house</h4>
                <ul class="detail-list">
                    <li><?php echo date('M d, Y', strtotime('+5 days')); ?>, 1:00&nbsp;–&nbsp;3:00&nbsp;PM</li>
                </ul>
            </div>

            <?php if ($rentEstimate): ?>
            <div class="detail-card">
                <h3>Rental listing details</h3>
                <p>Information for the long‑term rental version of this property.</p>
                <ul class="detail-list">
                    <li>Listing ID: <?php echo strtoupper($current['id']) . '-RENT'; ?></li>
                    <li>Status: Available for rent</li>
                    <li>Rent: $<?php echo number_format($rentEstimate); ?></li>
                    <li>Deposit: $<?php echo number_format($rentEstimate); ?></li>
                    <li>Application fee: $45</li>
                    <li>Lease terms: 12 months</li>
                    <li>Available from: <?php echo date('M d, Y', strtotime('+1 weeks')); ?></li>
                    <li>Days on market: <?php echo (int)$marketDaysOnMarket; ?></li>
                </ul>
            </div>
            <?php endif; ?>

            <div class="detail-card">
                <h3>Market data &amp; trends</h3>
                <div class="card-grid">
                    <div>
                        <h4>Sale market</h4>
                        <ul class="detail-list">
                            <li>Average sale price: $<?php echo number_format($marketSaleAvgPrice); ?></li>
                            <li>Median sale price: $<?php echo number_format($marketSaleMedianPrice); ?></li>
                            <li>Average price per sqft: $<?php echo number_format($marketPricePerSqFt); ?></li>
                            <li>Sales volume: <?php echo (int)$marketSalesVolume; ?></li>
                            <li>Sales count: <?php echo (int)$marketSaleCount; ?></li>
                            <li>Days on market: <?php echo (int)$marketDaysOnMarket; ?></li>
                            <li>Inventory: <?php echo (int)$marketInventory; ?></li>
                        </ul>
                    </div>
                    <div>
                        <h4>Rental market</h4>
                        <ul class="detail-list">
                            <li>Average rent: $<?php echo number_format($marketRentAvg); ?></li>
                            <li>Median rent: $<?php echo number_format($marketRentMedian); ?></li>
                            <li>Rent per sqft: $<?php echo number_format($marketRentPerSqFt); ?></li>
                            <li>Rental count: <?php echo (int)$marketRentCount; ?></li>
                            <li>Vacancy rate: <?php echo number_format($marketVacancyRate, 1); ?>%</li>
                        </ul>
                    </div>
                </div>
                <h4>Sale trends (last 3 months)</h4>
                <ul class="detail-list small">
                    <?php foreach ($saleTrends as $trend): ?>
                    <li><?php echo $trend['month'] . ' ' . $trend['year']; ?> – Median: $<?php echo number_format($trend['medianSalePrice']); ?>; Average: $<?php echo number_format($trend['averageSalePrice']); ?></li>
                    <?php endforeach; ?>
                </ul>
                <?php if (!empty($rentTrends)): ?>
                <h4>Rent trends (last 3 months)</h4>
                <ul class="detail-list small">
                    <?php foreach ($rentTrends as $trend): ?>
                    <li><?php echo $trend['month'] . ' ' . $trend['year']; ?> – Median: $<?php echo number_format($trend['medianRent']); ?>; Average: $<?php echo number_format($trend['averageRent']); ?></li>
                    <?php endforeach; ?>
                </ul>
                <?php endif; ?>
                <h4>Trend charts</h4>
                <div class="chart-container">
                    <h5>Sale price trends</h5>
                    <canvas id="saleChart" width="400" height="250"></canvas>
                </div>
                <?php if (!empty($rentTrends)): ?>
                <div class="chart-container">
                    <h5>Rent price trends</h5>
                    <canvas id="rentChart" width="400" height="250"></canvas>
                </div>
                <?php endif; ?>
            </div>
        </aside>
</div>
</div>

<script src="https://cdn.jsdelivr.net/npm/chart.js@3.9.1/dist/chart.min.js"></script>
<script>
// Trend data arrays passed from PHP
const saleTrendData = <?php echo json_encode($saleTrends); ?>;
const rentTrendData = <?php echo json_encode($rentTrends); ?>;

// Comparable sale and rent arrays removed; data displayed in tables instead.

function drawChart(ctx, labels, medianData, averageData, label1, label2) {
    new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [
                {
                    label: label1,
                    data: medianData,
                    borderColor: 'rgba(54, 162, 235, 1)',
                    backgroundColor: 'rgba(54, 162, 235, 0.1)',
                    tension: 0.3,
                    fill: true,
                },
                {
                    label: label2,
                    data: averageData,
                    borderColor: 'rgba(255, 99, 132, 1)',
                    backgroundColor: 'rgba(255, 99, 132, 0.1)',
                    tension: 0.3,
                    fill: true,
                }
            ]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                x: {
                    grid: {
                        display: false
                    }
                },
                y: {
                    beginAtZero: false,
                    grid: {
                        color: 'rgba(209, 213, 219, 0.3)'
                    }
                }
            },
            plugins: {
                legend: {
                    position: 'bottom',
                }
            }
        }
    });
}

window.addEventListener('DOMContentLoaded', () => {
    // Draw sale chart
    if (saleTrendData && saleTrendData.length && document.getElementById('saleChart')) {
        const labels = saleTrendData.map(item => item.month + ' ' + item.year);
        const median = saleTrendData.map(item => item.medianSalePrice);
        const average = saleTrendData.map(item => item.averageSalePrice);
        const ctx = document.getElementById('saleChart').getContext('2d');
        drawChart(ctx, labels, median, average, 'Median sale price', 'Average sale price');
    }
    // Draw rent chart
    if (rentTrendData && rentTrendData.length && document.getElementById('rentChart')) {
        const labelsR = rentTrendData.map(item => item.month + ' ' + item.year);
        const medianR = rentTrendData.map(item => item.medianRent);
        const averageR = rentTrendData.map(item => item.averageRent);
        const ctxR = document.getElementById('rentChart').getContext('2d');
        drawChart(ctxR, labelsR, medianR, averageR, 'Median rent', 'Average rent');
    }

    // Comparable bar charts removed. Comparable sales and rentals are now shown in tables.
    // Rent slider update
    const rentSlider = document.getElementById('rentSlider');
    const rentValue = document.getElementById('rentValue');
    if (rentSlider && rentValue) {
        rentSlider.addEventListener('input', () => {
            const value = parseInt(rentSlider.value, 10);
            rentValue.textContent = '$' + value.toLocaleString();
        });
    }
});
</script>

<?php
include __DIR__ . '/footer.php';